"""
メインモジュール

    「スピード！」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import os
import sys

from models.difficulty_level import DifficultyLevel

if hasattr(sys, "_MEIPASS"):
    os.environ["KIVY_NO_CONSOLELOG"] = "1"

from kivy.config import Config

Config.set("graphics", "width", 800)
Config.set("graphics", "height", 600)
Config.set("graphics", "minimum_width", 800)
Config.set("graphics", "minimum_height", 600)
Config.set("graphics", "resizable", False)
Config.set("input", "mouse", "mouse, disable_multitouch")
Config.set("kivy", "exit_on_escape", "0")
# Config.set("modules", "showborder", "")

import kivy  # noqa
import kivy.resources  # noqa
from kivy.app import App  # noqa
from kivy.core.text import DEFAULT_FONT, LabelBase  # noqa

import constants  # noqa
import views.root_widget as root_widget  # noqa
from models.game_result import GameResult  # noqa
from models.my_config import MyConfig  # noqa
from models.shared_vars import SharedVars  # noqa
from my_utilities import helper  # noqa
from my_utilities import json_io  # noqa


class MainApp(App):
    use_kivy_settings = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        kivy.resources.resource_add_path(helper.get_resource_path("fonts"))
        kivy.resources.resource_add_path(helper.get_resource_path("images"))
        kivy.resources.resource_add_path(helper.get_resource_path("sounds"))

        LabelBase.register(DEFAULT_FONT, "TakaoPGothic.ttf")

        self.title = constants.PRODUCT_NAME
        self.icon = "favicon.ico"

        self._config: MyConfig = self._get_config_values()
        self._sv: SharedVars = SharedVars()
        self._sv.bgm_volume = self._config.bgm_volume
        self._sv.difficulty_level = DifficultyLevel(self._config.difficulty_level)
        self._sv.suit_color = self._config.suit_color

    def build(self):
        root = root_widget.RootWidget(self._sv)
        return root

    def on_stop(self):
        """アプリを終了したとき。"""

        self._sv.game_result = GameResult.DRAW

        self._output_config_values()

    def _get_config_values(self) -> MyConfig:
        """
        設定ファイルからアプリの設定値を読み込み、MyConfig のインスタンスに
        格納して返す。ファイルが存在しない場合は、既定値が格納された MyConfig の
        インスタンスを返す。

        Returns:
            MyConfig: 設定値が格納された MyConfig のインスタンス。
        """

        file_path: str = constants.CONF_FILE_PATH

        try:
            my_config: MyConfig = json_io.load_dataclass(MyConfig(), file_path)

        except Exception:
            my_config = MyConfig()

        return my_config

    def _output_config_values(self) -> None:
        """アプリの設定値を設定ファイルに出力する。"""

        self._config.bgm_volume = self._sv.bgm_volume
        self._config.difficulty_level = self._sv.difficulty_level
        self._config.suit_color = self._sv.suit_color

        file_path: str = constants.CONF_FILE_PATH

        try:
            json_io.output_dataclass(self._config, file_path)

        except Exception:
            pass


if __name__ == "__main__":
    MainApp().run()
