"""
バージョン情報画面を定義したクラス。

    「ちょこっと時間計算」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import re
import webbrowser
from typing import Callable, Literal, override

from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.uix.screenmanager import Screen

import constants
from my_utilities import helper

Builder.load_string("""
<VersionScreen>:
    BoxLayout:
        orientation: 'vertical'

        AnchorLayout:
            size_hint_y: 0.5
            BoxLayout:
                size_hint: 0.8, 0.8
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.35
                    Image:
                        id: icon_image
                        source: "image_512.png"
                        fit_mode: 'contain'
                AnchorLayout:
                    size_hint_x: 0.65
                    BoxLayout:
                        orientation: 'vertical'
                        size_hint_y: 0.7
                        Label:
                            id: product_name_label
                            size_hint_y: 0.4
                            font_size: sp(44)
                        Label:
                            id: version_label
                            size_hint_y: 0.3
                            font_size: sp(22)
                        Label:
                            id: copyright_label
                            size_hint_y: 0.3
                            font_size: sp(18)
        AnchorLayout:
            size_hint_y: 0.4

            ScrollView:
                id: scroll_view
                size_hint: 0.90, 0.98
                do_scroll_y: True
                do_scroll_x: False
                bar_width: sp(16)
                scroll_type: ['bars', 'content']

                Label:
                    id: license_label
                    markup: True
                    font_size: sp(16)
                    color: 0.0, 0.0, 0.0, 1.0
                    size_hint_y: None
                    height: self.texture_size[1]
                    text_size: self.width, None
                    halign: 'left'
                    valign: 'top'
                    line_height: 1.5
                    padding: sp(12), sp(20)
                    canvas.before:
                        Color:
                            rgba: 0.99, 0.99, 0.99, 1.0
                        Rectangle:
                            pos: self.pos
                            size: self.size
        AnchorLayout:
            size_hint_y: 0.1
            Label:
                text: "元の画面に戻るには、白枠以外の場所をクリックしてください。"
                font_size: sp(16)
""")


class VersionScreen(Screen):
    """バージョン情報画面を定義したクラス。"""

    def __init__(self, to_main_screen: Callable[[], None], **kwargs) -> None:
        """バージョン情報画面を定義したクラスのインスタンスを作成して返す。

        Args:
            to_main_screen (Callable[[], None]): メイン画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._to_main_screen: Callable[[], None] = to_main_screen
        """メイン画面に遷移するための関数"""

        self.ids.product_name_label.text = constants.PRODUCT_NAME
        self.ids.version_label.text = f"Version {constants.VERSION}"
        self.ids.copyright_label.text = constants.COPYRIGHT

        self.ids.license_label.text = self._get_license_text()
        self.ids.license_label.bind(on_ref_press=self._open_link)  # type: ignore

    def _get_license_text(self) -> str:
        """ライセンスファイルのテキストを読み込んで返す。

        Returns:
            str: ライセンスのテキスト。
        """

        LICENSE_PATH: str = "./LICENSE.TXT"
        """LICENSE ファイルのパス"""

        license_path: str = helper.get_resource_path(LICENSE_PATH)
        text: str = ""

        try:
            with open(license_path, "r", encoding="utf-8") as file:
                text = file.read()
        except Exception:
            text = ""
        else:
            pattern: str = r"(https?://.+?)$"
            repl: str = "[ref=\\1][color=0000ff][u]\\1[/u][/color][/ref]"
            text = re.sub(pattern, repl, text, flags=re.MULTILINE)

        return text

    def _open_link(self, _, value: str) -> None:
        """URL のリンクを開く。

        Args:
            value (str): URL。
        """

        webbrowser.open(value)

    @override
    def on_touch_down(self, touch: MotionEvent) -> Literal[True, None]:
        """画面がクリックされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。

        Returns:
            Literal[True, None]: 次の画面に遷移した場合は True
        """

        result: Literal[True, None] = super().on_touch_down(touch)

        if not self.ids.scroll_view.collide_point(*touch.pos):
            self._to_main_screen()
            return True

        return result
