"""
ルートウィジェット

    「ちょこっとカウンター」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from kivy.core.window import Window
from kivy.lang import Builder
from kivy.uix.screenmanager import NoTransition, ScreenManager
from kivy.uix.widget import Widget

import constants
from main_screen import MainScreen
from my_config import MyConfig
from version_screen import VersionScreen

Builder.load_string("""
<RootWidget>:
    BoxLayout:
        orientation: 'vertical'
        size: root.size
        pos: root.pos
        ScreenManager:
            id: screen_manager
""")


class RootWidget(Widget):
    """ルートウィジェット。"""

    def __init__(
        self,
        config: MyConfig,
        **kwargs,
    ) -> None:
        """ルートウィジェットのインスタンスを作成して返す。"""

        super().__init__(**kwargs)

        self._config: MyConfig = config
        """アプリの設定値等を管理するクラスのインスタンス"""

        self._screen_manager: ScreenManager = self.ids.screen_manager
        """Screen を管理する ScreenManager のインスタンス"""

        self._create_screens()

        self._screen_manager.transition = NoTransition()
        self._screen_manager.current = "main_screen"

    def _create_screens(self) -> None:
        """各画面のインスタンスを作成して、ScreenManager に追加する。"""

        self._screen_manager.add_widget(
            VersionScreen(
                self._to_main_screen,
                name="version_screen",
            )
        )
        self._screen_manager.add_widget(
            MainScreen(
                self._to_version_screen,
                name="main_screen",
            )
        )

    def _to_version_screen(self) -> None:
        """バージョン情報画面に遷移する。"""

        self._config.window_pos_left = Window.left
        self._config.window_pos_top = Window.top
        Window.size = (800, 600)
        Window.left = 100
        Window.top = 100
        self._screen_manager.current = "version_screen"

    def _to_main_screen(self) -> None:
        """メイン画面に遷移する。"""

        Window.size = (constants.WIDTH, constants.HEIGHT)
        Window.left = self._config.window_pos_left
        Window.top = self._config.window_pos_top
        self._screen_manager.current = "main_screen"
