"""
メイン画面を定義したクラス。

    「ちょこっとカウンター」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from typing import Callable

from kivy.core.window import Window
from kivy.lang import Builder
from kivy.uix.screenmanager import Screen

Builder.load_string("""
<MainScreen>:
    BoxLayout:
        orientation: 'horizontal'
        size: root.size
        padding: sp(6)
        spacing: sp(4)
        Button:
            id: foreground_button
            text: '最前面'
            on_press: root._on_foreground_button_press()
            size_hint_x: 0.2
        Label:
            id: count_label
            size_hint_x: 0.25
            valign: 'middle'
            halign: 'right'
            text_size: self.size
            font_size: sp(19)
            padding_x: sp(4)
            color: 0, 0, 0, 1
            canvas.before:
                Color:
                    rgb: 1.0, 1.0, 1.0
                Rectangle:
                    pos: self.pos
                    size: self.size
        Button:
            id: count_up_button
            text: '+ 1'
            font_size: sp(18)
            size_hint_x: 0.2
            on_press: root._on_count_up_button_press()
        Button:
            id: count_down_button
            text: '- 1'
            font_size: sp(18)
            size_hint_x: 0.1
            on_press: root._on_count_down_button_press()
        Button:
            id: reset_button
            text: 'リセット'
            size_hint_x: 0.15
            on_press: root._on_reset_button_press()
        ImageButton:
            size_hint_x: 0.1
            on_press: root._on_version_info_button_press()

<ImageButton@ButtonBehavior+Image>:
    source: 'image_64.png'
    fit_mode: 'contain'
""")


class MainScreen(Screen):
    """メイン画面を定義したクラス。"""

    def __init__(self, to_version_screen: Callable[[], None], **kwargs) -> None:
        """
        メイン画面のインスタンスを作成して返す。

        Args:
            to_version_screen (Callable[[], None]): バージョン情報画面に遷移するための関数。
        """
        super().__init__(**kwargs)

        self._to_version_screen: Callable[[], None] = to_version_screen
        """バージョン情報画面に遷移するためのコールバック関数"""

        self._counter: int = 0
        """カウンターの値"""

        self.ids.count_label.text = f"{self._counter:,}"

        if Window.always_on_top:
            self.ids.foreground_button.text = "✓最前面"
        else:
            self.ids.foreground_button.text = "最前面"

    def _on_foreground_button_press(self, *args) -> None:
        """「最前面」ボタンがクリックされたときの処理。"""

        if Window.always_on_top:
            Window.always_on_top = False
            self.ids.foreground_button.text = "最前面"
        else:
            Window.always_on_top = True
            self.ids.foreground_button.text = "✓最前面"

    def _on_count_up_button_press(self, *args) -> None:
        """「+1」ボタンがクリックされたときの処理。"""

        self._counter += 1
        if self._counter == 1_000_000:
            self._counter = 0

        self.ids.count_label.text = f"{self._counter:,}"

    def _on_count_down_button_press(self, *args) -> None:
        """「-1」ボタンがクリックされたときの処理。"""

        self._counter -= 1
        if self._counter < 0:
            self._counter = 0

        self.ids.count_label.text = f"{self._counter:,}"

    def _on_reset_button_press(self, *args) -> None:
        """「リセット」ボタンがクリックされたときの処理。"""

        self._counter = 0
        self.ids.count_label.text = f"{self._counter:,}"

    def _on_version_info_button_press(self, *args) -> None:
        """「バージョン情報」ボタンがクリックされたときの処理。"""

        self._to_version_screen()
