"""
コントローラーモジュール。

    「ちょこっと色見本」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import colorsys


def normalize_rgb(rgb: list[int]) -> list[float]:
    """
    RGB値を正規化して返す。

    Args:
        rgb (list[int]): RGB 値のリスト。

    Returns:
        正規化されたRGB値のリスト。
    """

    return [rgb[0] / 255.0, rgb[1] / 255.0, rgb[2] / 255.0]


def rgb_to_hex_string(rgb: list[int]) -> str:
    """
    RGB 値を16進数表記の文字列に変換して返す。

    Args:
        rgb (list[int]): RGB値のリスト。

    Returns:
        16進数表記の文字列。
    """

    return f"#{rgb[0]:02X}{rgb[1]:02X}{rgb[2]:02X}"


def get_color_code_value(
    color_format: str, rgb: list[int], color_names: dict[str, str]
) -> str:
    """
    現在の RGB 値からカラーコードの文字列を取得する。

    Args:
        color_format (str): カラーフォーマット名。
        rgb (list[int]): RGB 値のリスト。
        color_names (dict[str, str]): 色名とその 16 進数表記の辞書。

    Returns:
        カラーコードの文字列。
    """

    normals: list[float] = normalize_rgb(rgb)
    hls_normals: tuple[float, float, float] = colorsys.rgb_to_hls(*normals)
    hsv_normals: tuple[float, float, float] = colorsys.rgb_to_hsv(*normals)

    match color_format:
        case "HEX":
            return rgb_to_hex_string(rgb)
        case "RGB":
            return ", ".join(str(c) for c in rgb)
        case "RGB\nNormal":
            return ", ".join(f"{n:.3f}" for n in normals)
        case "HLS":
            h: int = int(hls_normals[0] * 360)
            l: int = int(hls_normals[1] * 100)  # noqa: E741
            s: int = int(hls_normals[2] * 100)
            return f"{h}, {l}, {s}"
        case "HLS\nNormal":
            return ", ".join(f"{n:.3f}" for n in hls_normals)
        case "HSV":
            h: int = int(hsv_normals[0] * 360)
            s: int = int(hsv_normals[1] * 100)
            v: int = int(hsv_normals[2] * 100)
            return f"{h}, {s}, {v}"
        case "HSV\nNormal":
            return ", ".join(f"{n:.3f}" for n in hsv_normals)
        case "Name":
            for name, hex_code in color_names.items():
                rgb_from_name: list[int] | None = _get_rgb_from_hex_string(hex_code)
                if rgb_from_name == rgb:
                    return name

            return "(No Name)"
        case _:
            return ""


def get_rgb_from_color_code(
    color_code: str, color_format: str, color_names: dict[str, str]
) -> list[int] | None:
    """
    カラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。
        color_format (str): カラーフォーマット名。
        color_names (dict[str, str]): 色名とその 16 進数表記の辞書。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    match color_format:
        case "HEX":
            return _get_rgb_from_hex_string(color_code)
        case "RGB":
            return _get_rgb_from_rgb_string(color_code)
        case "RGB\nNormal":
            return _get_rgb_from_rgb_normal_string(color_code)
        case "HLS":
            return _get_rgb_from_hls_string(color_code)
        case "HLS\nNormal":
            return _get_rgb_from_his_normal_string(color_code)
        case "HSV":
            return _get_rgb_from_hsv_string(color_code)
        case "HSV\nNormal":
            return _get_rgb_from_hsv_normal_string(color_code)
        case "Name":
            return _get_rgb_from_name_string(color_code, color_names)
        case _:
            return None


def _get_rgb_from_hex_string(color_code: str) -> list[int] | None:
    """
    16 進数表記のカラーコードの文字列から RGB 値を取得する。
    Args:
        color_code (str): カラーコードの文字列。
    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        if not color_code.startswith("#") or len(color_code) != 7:
            return None

        r: int = int(color_code[1:3], 16)
        g: int = int(color_code[3:5], 16)
        b: int = int(color_code[5:7], 16)
        return [r, g, b]

    except ValueError:
        return None


def _get_rgb_from_rgb_string(color_code: str) -> list[int] | None:
    """
    "R, G, B" 形式のカラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        r: int = int(parts[0].strip())
        g: int = int(parts[1].strip())
        b: int = int(parts[2].strip())
        if r < 0 or r > 255 or g < 0 or g > 255 or b < 0 or b > 255:
            return None

    except (ValueError, IndexError):
        return None

    return [r, g, b]


def _get_rgb_from_rgb_normal_string(color_code: str) -> list[int] | None:
    """
    "R, G, B" 形式の正規化カラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        fr: float = float(parts[0].strip())
        fg: float = float(parts[1].strip())
        fb: float = float(parts[2].strip())
        if fr < 0.0 or fr > 1.0 or fg < 0.0 or fg > 1.0 or fb < 0.0 or fb > 1.0:
            return None

    except (ValueError, IndexError):
        return None

    return [int(fr * 255), int(fg * 255), int(fb * 255)]


def _get_rgb_from_hls_string(color_code: str) -> list[int] | None:
    """
    "H, L, S" 形式のカラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        h: float = float(parts[0].strip()) / 360.0
        l: float = float(parts[1].strip()) / 100.0  # noqa: E741
        s: float = float(parts[2].strip()) / 100.0
        if h < 0.0 or h > 1.0 or l < 0.0 or l > 1.0 or s < 0.0 or s > 1.0:
            return None

        r, g, b = colorsys.hls_to_rgb(h, l, s)
        return [int(r * 255), int(g * 255), int(b * 255)]

    except (ValueError, IndexError):
        return None


def _get_rgb_from_his_normal_string(color_code: str) -> list[int] | None:
    """
    "H, L, S" 形式の正規化カラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        fh: float = float(parts[0].strip())
        fl: float = float(parts[1].strip())
        fs: float = float(parts[2].strip())
        if fh < 0.0 or fh > 1.0 or fl < 0.0 or fl > 1.0 or fs < 0.0 or fs > 1.0:
            return None

        r, g, b = colorsys.hls_to_rgb(fh, fl, fs)
        return [int(r * 255), int(g * 255), int(b * 255)]

    except (ValueError, IndexError):
        return None


def _get_rgb_from_hsv_string(color_code: str) -> list[int] | None:
    """
    "H, S, V" 形式のカラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        h: float = float(parts[0].strip()) / 360.0
        s: float = float(parts[1].strip()) / 100.0
        v: float = float(parts[2].strip()) / 100.0
        if h < 0.0 or h > 1.0 or s < 0.0 or s > 1.0 or v < 0.0 or v > 1.0:
            return None

        r, g, b = colorsys.hsv_to_rgb(h, s, v)
        return [int(r * 255), int(g * 255), int(b * 255)]

    except (ValueError, IndexError):
        return None


def _get_rgb_from_hsv_normal_string(color_code: str) -> list[int] | None:
    """
    "H, S, V" 形式の正規化カラーコードの文字列から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    try:
        parts: list[str] = color_code.split(",")
        fh: float = float(parts[0].strip())
        fs: float = float(parts[1].strip())
        fv: float = float(parts[2].strip())
        if fh < 0.0 or fh > 1.0 or fs < 0.0 or fs > 1.0 or fv < 0.0 or fv > 1.0:
            return None

        r, g, b = colorsys.hsv_to_rgb(fh, fs, fv)
        return [int(r * 255), int(g * 255), int(b * 255)]

    except (ValueError, IndexError):
        return None


def _get_rgb_from_name_string(
    color_code: str, color_names: dict[str, str]
) -> list[int] | None:
    """
    色名から RGB 値を取得する。

    Args:
        color_code (str): カラーコードの文字列。
        color_names (dict[str, str]): 色名とその 16 進数表記の辞書。

    Returns:
        RGB 値のリスト。解析に失敗した場合は None を返す。
    """

    hex_code: str | None = color_names.get(color_code.lower())
    if hex_code is not None:
        return _get_rgb_from_hex_string(hex_code)
    else:
        return None


def get_color_names() -> dict[str, str]:
    """
    色名とその 16 進数表記の辞書を返す。

    Returns:
        色名と 16 進数表記の辞書。
    """

    return {
        "black": "#000000",
        "navy": "#000080",
        "darkblue": "#00008b",
        "mediumblue": "#0000cd",
        "blue": "#0000ff",
        "darkgreen": "#006400",
        "green": "#008000",
        "teal": "#008080",
        "darkcyan": "#008b8b",
        "deepskyblue": "#00bfff",
        "darkturquoise": "#00ced1",
        "mediumspringgreen": "#00fa9a",
        "lime": "#00ff00",
        "springgreen": "#00ff7f",
        "aqua": "#00ffff",
        "cyan": "#00ffff",
        "midnightblue": "#191970",
        "dodgerblue": "#1e90ff",
        "lightseagreen": "#20b2aa",
        "forestgreen": "#228b22",
        "seagreen": "#2e8b57",
        "darkslategray": "#2f4f4f",
        "limegreen": "#32cd32",
        "mediumseagreen": "#3cb371",
        "turquoise": "#40e0d0",
        "royalblue": "#4169e1",
        "steelblue": "#4682b4",
        "darkslateblue": "#483d8b",
        "mediumturquoise": "#48d1cc",
        "indigo": "#4b0082",
        "darkolivegreen": "#556b2f",
        "cadetblue": "#5f9ea0",
        "cornflowerblue": "#6495ed",
        "mediumaquamarine": "#66cdaa",
        "dimgray": "#696969",
        "slateblue": "#6a5acd",
        "olivedrab": "#6b8e23",
        "slategray": "#708090",
        "lightslategray": "#778899",
        "mediumslateblue": "#7b68ee",
        "lawngreen": "#7cfc00",
        "chartreuse": "#7fff00",
        "aquamarine": "#7fffd4",
        "maroon": "#800000",
        "purple": "#800080",
        "olive": "#808000",
        "gray": "#808080",
        "skyblue": "#87ceeb",
        "lightskyblue": "#87cefa",
        "blueviolet": "#8a2be2",
        "darkred": "#8b0000",
        "darkmagenta": "#8b008b",
        "saddlebrown": "#8b4513",
        "darkseagreen": "#8fbc8f",
        "lightgreen": "#90ee90",
        "mediumpurple": "#9370db",
        "darkviolet": "#9400d3",
        "palegreen": "#98fb98",
        "darkorchid": "#9932cc",
        "yellowgreen": "#9acd32",
        "sienna": "#a0522d",
        "brown": "#a52a2a",
        "darkgray": "#a9a9a9",
        "lightblue": "#add8e6",
        "greenyellow": "#adff2f",
        "paleturquoise": "#afeeee",
        "lightsteelblue": "#b0c4de",
        "powderblue": "#b0e0e6",
        "firebrick": "#b22222",
        "darkgoldenrod": "#b8860b",
        "mediumorchid": "#ba55d3",
        "rosybrown": "#bc8f8f",
        "darkkhaki": "#bdb76b",
        "silver": "#c0c0c0",
        "mediumvioletred": "#c71585",
        "indianred": "#cd5c5c",
        "peru": "#cd853f",
        "chocolate": "#d2691e",
        "tan": "#d2b48c",
        "lightgrey": "#d3d3d3",
        "thistle": "#d8bfd8",
        "orchid": "#da70d6",
        "goldenrod": "#daa520",
        "palevioletred": "#db7093",
        "crimson": "#dc143c",
        "gainsboro": "#dcdcdc",
        "plum": "#dda0dd",
        "burlywood": "#deb887",
        "lightcyan": "#e0ffff",
        "lavender": "#e6e6fa",
        "darksalmon": "#e9967a",
        "violet": "#ee82ee",
        "palegoldenrod": "#eee8aa",
        "lightcoral": "#f08080",
        "khaki": "#f0e68c",
        "aliceblue": "#f0f8ff",
        "honeydew": "#f0fff0",
        "azure": "#f0ffff",
        "sandybrown": "#f4a460",
        "wheat": "#f5deb3",
        "beige": "#f5f5dc",
        "whitesmoke": "#f5f5f5",
        "mintcream": "#f5fffa",
        "ghostwhite": "#f8f8ff",
        "salmon": "#fa8072",
        "antiquewhite": "#faebd7",
        "linen": "#faf0e6",
        "lightgoldenrodyellow": "#fafad2",
        "oldlace": "#fdf5e6",
        "red": "#ff0000",
        "fuchsia": "#ff00ff",
        "magenta": "#ff00ff",
        "deeppink": "#ff1493",
        "orangered": "#ff4500",
        "tomato": "#ff6347",
        "hotpink": "#ff69b4",
        "coral": "#ff7f50",
        "darkorange": "#ff8c00",
        "lightsalmon": "#ffa07a",
        "orange": "#ffa500",
        "lightpink": "#ffb6c1",
        "pink": "#ffc0cb",
        "gold": "#ffd700",
        "peachpuff": "#ffdab9",
        "navajowhite": "#ffdead",
        "moccasin": "#ffe4b5",
        "bisque": "#ffe4c4",
        "mistyrose": "#ffe4e1",
        "blanchedalmond": "#ffebcd",
        "papayawhip": "#ffefd5",
        "lavenderblush": "#fff0f5",
        "seashell": "#fff5ee",
        "cornsilk": "#fff8dc",
        "lemonchiffon": "#fffacd",
        "floralwhite": "#fffaf0",
        "snow": "#fffafa",
        "yellow": "#ffff00",
        "lightyellow": "#ffffe0",
        "ivory": "#fffff0",
        "white": "#ffffff",
    }
