"""
ルートウィジェットを定義したクラス

    「ロト6 当せん数字予測アプリ」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from kivy.lang import Builder
from kivy.uix.screenmanager import ScreenManager, SlideTransition
from kivy.uix.widget import Widget

from shared_vars import SharedVars
from views.ball_sets_screen import BallSetsScreen
from views.by_ball_set_screen import ByBallSetScreen
from views.main_screen import MainScreen
from views.my_actionbar import MyActionBar
from views.version_info_screen import VersionInfoScreen

Builder.load_string("""
<RootWidget>:
    BoxLayout:
        orientation: 'vertical'
        size: root.size
        MyActionBar:
            id: actionbar
        ScreenManager:
            id: screen_manager
""")


class RootWidget(Widget):
    """ルートウィジェットを定義したクラス"""

    def __init__(self, shared_vars: SharedVars, **kwargs):
        """ルートウィジェットを定義したクラスのインスタンスを作成して返す。

        Args:
            shared_vars: (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
        """

        super().__init__(**kwargs)

        self._shared_vars: SharedVars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._screen_manager: ScreenManager = self.ids.screen_manager
        """画面遷移を管理する ScreenManager のインスタンス"""

        self._actionbar: MyActionBar = self.ids.actionbar
        """アクションバーのインスタンス"""

        self._prev_screen: str = "main_screen"
        """前の画面の名前"""

        self._create_screens()
        self._toggle_menus_state()

    def _create_screens(self) -> None:
        """各画面のインスタンスを作成して、ScreenManager に追加する。"""

        self._main_screen = MainScreen(self._shared_vars, name="main_screen")
        self._screen_manager.add_widget(self._main_screen)
        self._screen_manager.add_widget(
            BallSetsScreen(self._shared_vars, name="ball_sets_screen")
        )
        self._screen_manager.add_widget(
            ByBallSetScreen(self._shared_vars, name="by_ball_set_screen")
        )
        self._screen_manager.add_widget(VersionInfoScreen(name="version_info_screen"))

    def on_previous_icon_click(self):
        """アクションバー左端のアイコンをクリックしたとき。"""

        if self._screen_manager.current == "version_info_screen":
            self._screen_manager.transition = SlideTransition(direction="right")
            self._screen_manager.current = self._prev_screen
        elif self._screen_manager.current:
            self._prev_screen = self._screen_manager.current
            self._screen_manager.transition = SlideTransition(direction="left")
            self._screen_manager.current = "version_info_screen"

        self._toggle_menus_state()

    def _toggle_menus_state(self) -> None:
        """状況に応じてアクションバーの各項目の有効・無効を切り替える。"""

        if self._screen_manager.current == "main_screen":
            self._actionbar.main_screen_button.disabled = True
            self._actionbar.ball_set_button.disabled = False
            self._actionbar.occurrence_rate_button.disabled = False
            self._actionbar.write_csv_button.disabled = False
        elif self._screen_manager.current == "ball_sets_screen":
            self._actionbar.main_screen_button.disabled = False
            self._actionbar.ball_set_button.disabled = True
            self._actionbar.occurrence_rate_button.disabled = False
            self._actionbar.write_csv_button.disabled = True
        elif self._screen_manager.current == "by_ball_set_screen":
            self._actionbar.main_screen_button.disabled = False
            self._actionbar.ball_set_button.disabled = False
            self._actionbar.occurrence_rate_button.disabled = True
            self._actionbar.write_csv_button.disabled = True
        else:
            self._actionbar.main_screen_button.disabled = False
            self._actionbar.ball_set_button.disabled = False
            self._actionbar.occurrence_rate_button.disabled = False
            self._actionbar.write_csv_button.disabled = True

    def on_main_screen_button_click(self) -> None:
        """アクションバーの「データ管理画面」がクリックされたとき。"""

        self._screen_manager.transition = SlideTransition(direction="up")
        self._screen_manager.current = "main_screen"
        self._toggle_menus_state()

    def on_ball_sets_info_button_click(self) -> None:
        """アクションバーの「セット球情報」がクリックされたとき。"""

        self._screen_manager.transition = SlideTransition(direction="down")
        self._screen_manager.current = "ball_sets_screen"
        self._toggle_menus_state()

    def on_by_ball_set_button_click(self) -> None:
        """アクションバーの「セット球別集計」がクリックされたとき。"""

        self._screen_manager.transition = SlideTransition(direction="down")
        self._screen_manager.current = "by_ball_set_screen"
        self._toggle_menus_state()

    def on_write_csv_button_click(self) -> None:
        """アクションバーの「CSV ファイル出力」がクリックされたとき。"""

        self._main_screen.show_save_file_dialog()
