"""
最近のセット球出現状況のデータグリッドのレコード行とヘッダー行の定義

    「ロト6 当せん数字予測アプリ」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from kivy.lang.builder import Builder
from kivy.properties import (
    ListProperty,
    NumericProperty,
    ObjectProperty,
    StringProperty,
)

from datagrid.data_label import DataLabel
from datagrid.header_item import HeaderItem
from datagrid.header_label import HeaderLabel
from datagrid.row_item import RowItem

Builder.load_string("""
<DataGridRecentlyRow>:
    selectable: False

    DataLabel:
        text: str(root.times)
        size_hint_x: 0.5
        text_size: self.size
        halign: 'right'
        current_bg_color: root.bg_color
    DataLabel:
        text: root.ball_set
        size_hint_x: 0.5
        text_size: self.size
        halign: 'center'
        current_bg_color: root.bg_color

<DataGridRecentlyHeader>:
    HeaderLabel:
        text: root.times
        size_hint_x: 0.5
        text_size: self.size
        halign: 'center'
    HeaderLabel:
        text: root.ball_set
        size_hint_x: 0.5
        text_size: self.size
        halign: 'center'
""")


class DataGridRecentlyRow(RowItem):
    """最近のセット球選出状況のデータグリッドのレコード行を定義したクラス"""

    times: int = NumericProperty(0)
    """抽選回"""

    ball_set: str = StringProperty("")
    """セット球の名称"""

    is_duplicated: bool = ObjectProperty(False)
    """重複があれば True"""

    bg_color: list[float] = ListProperty([1.0, 1.0, 1.0, 1.0])
    """背景色"""

    def __init__(self, **kwargs):
        """最近のセット球選出状況のデータグリッドのレコード行のインスタンスを作成して返す。

        Args:
            **kwargs: キーワード引数
        """

        super().__init__(**kwargs)

        self.bind(is_duplicated=self._set_bg_color)  # type: ignore

    def _set_bg_color(self, instance, value) -> None:
        """背景色を変更する。"""

        if self.is_duplicated:
            self.bg_color = [1.0, 0.9, 0.9, 1.0]
        else:
            self.bg_color = [1.0, 1.0, 1.0, 1.0]


class DataGridRecentlyHeader(HeaderItem):
    """最近のセット球選出状況のデータグリッドのヘッダー行を定義したクラス"""

    times: str = StringProperty("抽選回")
    """抽選回"""

    ball_set: str = StringProperty("セット球")
    """セット球の名称"""
