"""
レコード行のレイアウトの定義

    Last modified: 2025/06/24

    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from typing import Literal, override

from datagrid.data_label import DataLabel
from kivy.input.motionevent import MotionEvent
from kivy.properties import BooleanProperty
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.recycleview.views import RecycleDataViewBehavior


class RowItem(BoxLayout, RecycleDataViewBehavior):
    """レコード行のレイアウトを定義したクラス"""

    selectable: bool = BooleanProperty(False)
    """選択可能であれば True"""

    def __init__(self, is_dummy: bool = False, **kwargs):
        """レコード行のレイアウトのインスタンスを作成して返す。

        Args:
            is_dummy (bool): ダミー行であれば True。デフォルトは False。
        """

        super().__init__(**kwargs)

        self.is_dummy = is_dummy
        """ダミー行であれば True"""

        self._index: int = -1
        """この行のインデックス"""

        self.orientation = "horizontal"

    @override
    def refresh_view_attrs(self, rv, index, data) -> None:
        """Catch and handle the view changes"""

        if self.selectable:
            self._update_selection_state(False)

        self._index = index
        super().refresh_view_attrs(rv, index, data)

    @override
    def on_touch_down(self, touch: MotionEvent) -> Literal[True] | None:
        """Add selection on touch down"""

        if super().on_touch_down(touch):
            return True

        if self.collide_point(*touch.pos):
            return self.parent.select_with_touch(self._index, touch)

    @override
    def apply_selection(self, rv, index, is_selected: bool) -> None:
        """Respond to the selection of items in the view."""

        if is_selected:
            if self.parent.selected_row is self:
                self.parent.selected_row = RowItem(is_dummy=True)
                if self.selectable:
                    self._update_selection_state(False)
                return
            else:
                self.parent.selected_row = self

        if self.selectable:
            self._update_selection_state(is_selected)

    def _update_selection_state(self, is_selected: bool) -> None:
        """行の選択状態を変更する。

        Args:
            is_selected (bool): 選択されていれば True。
        """

        for child in self.children:
            if isinstance(child, DataLabel):
                child.is_selected = is_selected
