"""
バージョン情報画面を定義したクラス

    「ロト6 当せん数字予測アプリ」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import re
import webbrowser

from kivy.lang import Builder
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen

import constants
import helper

Builder.load_string("""
<VersionInfoScreen>:
    label_name: label_name
    label_version: label_version
    label_copyright: label_copyright
    label_license: label_license

    AnchorLayout:
        anchor_x: 'center'
        anchor_y: 'center'
        BoxLayout:
            orientation: 'vertical'
            size_hint: 0.9, 0.9
            spacing: dp(20)
            BoxLayout:
                size_hint_y: 0.5
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.35
                    Image:
                        source: 'image_256.png'
                        fit_mode: 'contain'
                        size_hint_x: 0.8
                AnchorLayout:
                    size_hint_x: 0.65
                    BoxLayout:
                        orientation: 'vertical'
                        size_hint: 0.9, 0.7
                        Label:
                            id: label_name
                            size_hint_y: 0.4
                            font_size: self.width/13
                        Label:
                            id: label_version
                            size_hint_y: 0.3
                            font_size: self.width/15
                        Label:
                            id: label_copyright
                            size_hint_y: 0.3
                            font_size: self.width/20
            AnchorLayout:
                size_hint_y: 0.5
                anchor_x: 'center'
                anchor_y: 'center'

                ScrollView:
                    scroll_type: ['bars', 'content']
                    bar_width: dp(16)
                    do_scroll_y: True
                    do_scroll_x: False
                    Label:
                        id: label_license
                        markup: True
                        font_size: self.width/49
                        color: 0.95, 0.95, 0.95, 1.0
                        size_hint_y: None
                        height: self.texture_size[1]
                        text_size: self.width, None
                        halign: 'left'
                        valign: 'top'
                        line_height: 1.5
                        padding: dp(12), dp(20)
                        canvas.before:
                            Color:
                                rgba: 0.13, 0.13, 0.13, 1.0
                            Rectangle:
                                pos: self.pos
                                size: self.size
""")


class VersionInfoScreen(Screen):
    """バージョン情報画面を定義したクラス。"""

    def __init__(self, **kwargs) -> None:
        """バージョン情報画面を定義したクラスのインスタンスを作成して返す。"""

        super().__init__(**kwargs)

        self.ids.label_name.text = constants.PRODUCT_NAME
        self.ids.label_version.text = f"Version {constants.VERSION}"
        self.ids.label_copyright.text = constants.COPYRIGHT

        self._label_license: Label = self.ids.label_license
        self._label_license.text = self._get_license_text()
        self._label_license.bind(on_ref_press=self._open_link)  # type: ignore

    def _get_license_text(self) -> str:
        """ライセンスファイルのテキストを読み込んで返す。

        Returns:
            str: ライセンスのテキスト。
        """

        license_path: str = helper.get_resource_path("LICENSE.TXT")
        text: str = ""

        try:
            with open(license_path, "r", encoding="utf-8") as file:
                text = file.read()
        except Exception:
            text = ""
        else:
            pattern: str = r"(https?://.+?)$"
            repl: str = "[ref=\\1][color=b8a0ff][u]\\1[/u][/color][/ref]"
            text = re.sub(pattern, repl, text, flags=re.MULTILINE)

        return text

    def _open_link(self, _, value: str) -> None:
        """URL のリンクを開く。

        Args:
            value (str): URL。
        """

        webbrowser.open(value)
