"""
数字別の出現状況のデータグリッドのレコード行とヘッダー行の定義

    「ロト6 当せん数字予測アプリ」
    Copyright (c) 2025 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from kivy.lang.builder import Builder
from kivy.properties import (
    NumericProperty,
    StringProperty,
)

from datagrid.data_label import DataLabel
from datagrid.header_item import HeaderItem
from datagrid.header_label import HeaderLabel
from datagrid.row_item import RowItem

Builder.load_string("""
<DataGridFrequencyCountRow>:
    label_number: label_number
    label_count: label_count
    label_rate: label_rate
    label_cdf_value: label_cdf_value

    selectable: True

    DataLabel:
        id: label_number
        text: f"{root.numbers :0>2}"
        size_hint_x: 0.18
        text_size: self.size
        halign: 'center'
    DataLabel:
        id: label_count
        text: str(root.count)
        size_hint_x: 0.22
        text_size: self.size
        halign: 'right'
    DataLabel:
        id: label_rate
        text: f"{root.rate :.1%}"
        size_hint_x: 0.2
        text_size: self.size
        halign: 'right'
    DataLabel:
        id: label_cdf_value
        text: f"{root.cdf_value :.5f}"
        size_hint_x: 0.4
        text_size: self.size
        halign: 'right'

<DataGridFrequencyCountHeader>:
    HeaderLabel:
        text: root.numbers
        size_hint_x: 0.18
        text_size: self.size
        halign: 'center'
    HeaderLabel:
        text: root.count
        size_hint_x: 0.22
        text_size: self.size
        halign: 'center'
    HeaderLabel:
        text: root.rate
        size_hint_x: 0.2
        text_size: self.size
        halign: 'center'
    HeaderLabel:
        text: root.cdf_value
        size_hint_x: 0.4
        text_size: self.size
        halign: 'center'
""")


class DataGridFrequencyCountRow(RowItem):
    """数字別の出現回数のデータグリッドのレコード行を定義したクラス"""

    numbers: int = NumericProperty(0)
    """数字"""

    count: int = NumericProperty(0)
    """数字の出現回数"""

    rate: float = NumericProperty(0.0)
    """数字の出現率"""

    cdf_value: float = NumericProperty(0.0)
    """累積分布関数の値"""

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self._label_number: DataLabel = self.ids.label_number
        """数字のラベル"""

        self._label_count: DataLabel = self.ids.label_count
        """出現回数のラベル"""

        self._label_rate: DataLabel = self.ids.label_rate
        """出現率のラベル"""

        self._label_cdf_value: DataLabel = self.ids.label_cdf_value
        """累積分布関数の値のラベル"""

        self.bind(cdf_value=self._set_bg_color)  # type: ignore

    def _set_bg_color(self, instance, value) -> None:
        """背景色を変更する。"""

        if self.cdf_value > 0.975:
            self._label_number.default_bg_color = [0.92, 0.85, 1.0, 1.0]
            self._label_number.current_bg_color = self._label_number.default_bg_color
            self._label_count.default_bg_color = [1.0, 0.85, 0.85, 1.0]
            self._label_count.current_bg_color = self._label_count.default_bg_color
            self._label_rate.default_bg_color = [1.0, 0.85, 0.85, 1.0]
            self._label_rate.current_bg_color = self._label_rate.default_bg_color
            self._label_cdf_value.default_bg_color = [1.0, 0.85, 0.85, 1.0]
            self._label_cdf_value.current_bg_color = (
                self._label_cdf_value.default_bg_color
            )
        else:
            self._label_number.default_bg_color = [0.92, 0.92, 1.0, 1.0]
            self._label_number.current_bg_color = self._label_number.default_bg_color
            self._label_count.default_bg_color = [1.0, 1.0, 1.0, 1.0]
            self._label_count.current_bg_color = self._label_count.default_bg_color
            self._label_rate.default_bg_color = [1.0, 1.0, 1.0, 1.0]
            self._label_rate.current_bg_color = self._label_rate.default_bg_color
            self._label_cdf_value.default_bg_color = [1.0, 1.0, 1.0, 1.0]
            self._label_cdf_value.current_bg_color = (
                self._label_cdf_value.default_bg_color
            )


class DataGridFrequencyCountHeader(HeaderItem):
    """数字別の出現回数のデータグリッドのヘッダー行を定義したクラス"""

    numbers: str = StringProperty("数字")
    """数字"""

    count: str = StringProperty("出現回数")
    """数字の出現回数"""

    rate: str = StringProperty("出現率")
    """数字の出現率"""

    cdf_value: str = StringProperty("累積分布関数の値")
    """累積分布関数の値"""
