"""
バージョン情報画面を定義したクラス。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import re
import webbrowser
from typing import Callable, override

from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen
from kivy.uix.scrollview import ScrollView

import constants
from my_utilities import helper

Builder.load_string("""
<VersionScreen>:
    BoxLayout:
        orientation: 'vertical'
        canvas.before:
            Color:
                rgba: 144/256, 238/256, 144/256, 1.0
            Rectangle:
                pos: self.pos
                size: self.size

        AnchorLayout:
            size_hint_y: 0.5
            BoxLayout:
                size_hint: 0.8, 0.8
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.35
                    Image:
                        id: icon_image
                        fit_mode: 'contain'
                AnchorLayout:
                    size_hint_x: 0.65
                    BoxLayout:
                        orientation: 'vertical'
                        size_hint_y: 0.8
                        Label:
                            id: name_label
                            size_hint_y: 0.4
                            font_size: self.width/7
                            color: 0.0, 0.3, 0.0, 1.0
                        Label:
                            id: version_label
                            size_hint_y: 0.3
                            font_size: self.width/15
                            color: 0.0, 0.3, 0.0, 1.0
                        Label:
                            id: copyright_label
                            size_hint_y: 0.3
                            font_size: self.width/20
                            color: 0.0, 0.3, 0.0, 1.0
        AnchorLayout:
            size_hint_y: 0.4

            ScrollView:
                id: scroll_view
                size_hint: 0.90, 0.98
                do_scroll_y: True
                do_scroll_x: False
                bar_width: 16
                scroll_type: ['bars', 'content']

                Label:
                    id: license_label
                    markup: True
                    font_size: self.width/49
                    color: 0.0, 0.0, 0.0, 1.0
                    size_hint_y: None
                    height: self.texture_size[1]
                    text_size: self.width, None
                    halign: 'left'
                    valign: 'top'
                    line_height: 1.5
                    padding: 12, 20
                    canvas.before:
                        Color:
                            rgba: 0.99, 0.99, 0.99, 1.0
                        Rectangle:
                            pos: self.pos
                            size: self.size
        AnchorLayout:
            size_hint_y: 0.1
            Label:
                text: "元の画面に戻るには、白枠以外の場所をクリックしてください。"
                color: 0.0, 0.3, 0.0, 1.0
""")


class VersionScreen(Screen):
    """バージョン情報画面を定義したクラス。"""

    def __init__(self, to_next_screen: Callable[[], None], **kwargs) -> None:
        """バージョン情報画面を定義したクラスのインスタンスを作成して返す。

        Args:
            to_next_screen (Callable[[], None]): 次の画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._to_next_screen: Callable[[], None] = to_next_screen
        """次の画面に遷移するための関数"""

        self._icon_image: Image = self.ids.icon_image
        self._name_label: Label = self.ids.name_label
        self._version_label: Label = self.ids.version_label
        self._copyright_label: Label = self.ids.copyright_label
        self._license_label: Label = self.ids.license_label
        self._scroll_view: ScrollView = self.ids.scroll_view

        self._icon_image.source = "dam.png"
        self._name_label.text = constants.PRODUCT_NAME
        self._version_label.text = f"Version {constants.VERSION}"
        self._copyright_label.text = constants.COPYRIGHT

        self._license_label.text = self._get_license_text()
        self._license_label.bind(on_ref_press=self._open_link)  # type: ignore

    def _get_license_text(self) -> str:
        """ライセンスファイルのテキストを読み込んで返す。

        Returns:
            str: ライセンスのテキスト。
        """

        LICENSE_PATH: str = "./LICENSE.TXT"
        """LICENSE ファイルのパス"""

        license_path: str = helper.get_resource_path(LICENSE_PATH)
        text: str = ""

        try:
            with open(license_path, "r", encoding="utf-8") as file:
                text = file.read()
        except Exception:
            text = ""
        else:
            pattern: str = r"(https?://.+?)$"
            repl: str = "[ref=\\1][color=0000ff][u]\\1[/u][/color][/ref]"
            text = re.sub(pattern, repl, text, flags=re.MULTILINE)

        return text

    def _open_link(self, _, value: str) -> None:
        """URL のリンクを開く。

        Args:
            value (str): URL。
        """

        webbrowser.open(value)

    @override
    def on_touch_down(self, touch: MotionEvent) -> None:
        """画面がクリックされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        super().on_touch_down(touch)

        is_collision: bool = self._scroll_view.collide_point(*touch.pos)
        if (touch.button == "left") and (not is_collision):
            self._to_next_screen()
