"""
タイトル画面を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import webbrowser
from typing import Callable, override

from kivy.core.audio import Sound
from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.properties import StringProperty
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen

import constants
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene

Builder.load_string("""
<TitleScreen>:
    FloatLayout:
        canvas:
            Rectangle:
                source: "title_screen_bg.png"
                pos: self.pos
                size: self.size
        Label:
            text: root.copy_right
            font_size: dp(18)
            color: 0.3, 0.0, 0.0, 1.0
            pos_hint: {'center_x': 0.5, 'center_y': 0.72}
            size_hint: 1.0, 0.1
        Label:
            text: "画面をクリックしてください"
            font_size: dp(20)
            color: 0.0, 0.3, 0.0, 1.0
            pos_hint: {'center_x': 0.5, 'center_y': 0.14}
            size_hint: 0.8, 0.1
        Label:
            id: bgm_info_label
            markup: True
            font_size: dp(16)
            color: 0.6, 0.3, 0.0, 1.0
            pos_hint: {'center_x': 0.5, 'center_y': 0.042}
            size_hint: 0.8, 0.05
""")


class TitleScreen(Screen):
    """タイトル画面を管理するクラス。"""

    copy_right: str = StringProperty(constants.COPYRIGHT)
    """著作権表示"""

    def __init__(
        self, shared_vars: SharedVars, to_next_screen: Callable[[], None], **kwargs
    ) -> None:
        """タイトル画面を管理するクラスのインスタンスを作成して返す。

        Args:
            shared_vars (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
            to_next_screen (Callable[[], None]): 次の画面に遷移する関数。
        """

        super().__init__(**kwargs)

        self._shared_vars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._to_next_screen: Callable[[], None] = to_next_screen
        """次の画面に遷移する関数"""

        self._bgm_info_label: Label = self.ids.bgm_info_label
        """BGM の情報を表示するラベル"""

        self._bgm_info_label.bind(on_ref_press=self._open_link)  # type: ignore

    def _open_link(self, _, url: str) -> None:
        """URL のリンクを開く。

        Args:
            url (str): URL。
        """

        webbrowser.open(url)

    @override
    def on_pre_enter(self, *_) -> None:
        """画面が表示される前の処理。"""

        super().on_pre_enter()

        self._play_title_bgm()

        title: str = SoundScene.OPENING.title
        site_name: str = SoundScene.OPENING.site_name
        url: str = SoundScene.OPENING.site_url
        link: str = f"[color=3000ff][u][ref={url}]{url}[/ref][/u][/color]"
        self._bgm_info_label.text = f"音楽: 『{title}』（{site_name}: {link}）"

    def _play_title_bgm(self) -> None:
        """タイトル画面の BGM を再生する。"""

        scenes: list[SoundScene] = [
            SoundScene.NORMAL_MODE,
            SoundScene.WINNING_EXIT,
            SoundScene.DRAW_EXIT,
            SoundScene.LOSE_EXIT,
            SoundScene.DROP_OUT_EXIT,
        ]

        for scene in scenes:
            sound: Sound = self._shared_vars.sounds[scene]
            if sound.state == "play":
                sound.stop()

        bgm: Sound = self._shared_vars.sounds[SoundScene.OPENING]
        if bgm.state != "play":
            bgm.play()

    @override
    def on_touch_down(self, touch: MotionEvent) -> None:
        """画面をクリックしたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        super().on_touch_down(touch)

        is_collide: bool = self._bgm_info_label.collide_point(*touch.pos)

        if (touch.button == "left") and (not is_collide):
            self._to_next_screen()
