"""
ルートウィジェット

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging

from kivy.core.audio import SoundLoader
from kivy.lang import Builder
from kivy.uix.screenmanager import ScreenManager
from kivy.uix.widget import Widget

from models.game_scene import GameScene
from models.my_config import MyConfig
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene
from views.cards_screen import CardsScreen
from views.game_screen import GameScreen
from views.menu_screen import MenuScreen
from views.results_screen import ResultsScreen
from views.title_screen import TitleScreen
from views.version_screen import VersionScreen

Builder.load_string("""
<RootWidget>:
    BoxLayout:
        orientation: 'vertical'
        size: root.size
        ScreenManager:
            id: screen_manager
""")


class RootWidget(Widget):
    """ルートウィジェット。"""

    def __init__(
        self,
        config: MyConfig,
        shared_vars: SharedVars,
        logger: logging.Logger,
        **kwargs,
    ) -> None:
        """ルートウィジェットのインスタンスを作成して返す。"""

        super().__init__(**kwargs)

        self._config: MyConfig = config
        """アプリの設定値を監理するクラスのインスタンス"""

        self._shared_vars: SharedVars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

        self._screen_manager: ScreenManager = self.ids.screen_manager
        """画面遷移を管理する ScreenManager のインスタンス"""

        self._load_sounds()
        self._create_screens()

        self._screen_manager.current = "title_screen"

    def _create_screens(self) -> None:
        """各画面のインスタンスを作成して、ScreenManager に追加する。"""

        self._screen_manager.add_widget(
            TitleScreen(self._shared_vars, self._to_next_screen, name="title_screen")
        )
        self._screen_manager.add_widget(
            MenuScreen(
                self._config,
                self._shared_vars,
                self._to_next_screen,
                self._to_results_screen_from_menu,
                self._to_version_screen,
                name="menu_screen",
            )
        )
        self._screen_manager.add_widget(
            ResultsScreen(
                self._config,
                self._shared_vars,
                self._to_next_screen,
                name="results_screen",
            )
        )
        self._screen_manager.add_widget(
            VersionScreen(
                self._to_next_screen,
                name="version_screen",
            )
        )
        self._screen_manager.add_widget(
            GameScreen(
                self._config,
                self._shared_vars,
                self._logger,
                self._to_next_screen,
                lambda s: self._to_cards_screen(s),
                lambda s: self._to_title_screen_from_game(s),
                name="game_screen",
            )
        )
        self._screen_manager.add_widget(
            CardsScreen(
                self._config,
                self._shared_vars,
                lambda s: self._to_game_screen_from_card_list(s),
                name="cards_screen",
            )
        )

    def _load_sounds(self) -> None:
        """サウンドファイルを読み込む。"""

        for scene in SoundScene:
            if (sound := SoundLoader.load(scene.filename)) is not None:
                sound.volume = self._get_volume(scene)
                sound.loop = scene.is_bgm
                self._shared_vars.sounds[scene] = sound

    def _get_volume(self, scene: SoundScene) -> float:
        """個別に調整された音量を返す。

        Args:
            scene (SoundScene): BGM / 効果音の再生シーン。

        Returns:
            float: 個別に調整された音量。
        """

        volume: float = self._config.bgm_volume * scene.volume_scale
        volume = 100.0 if volume > 100.0 else volume

        return volume / 100.0

    ###########################################################################
    # 画面遷移

    def _to_cards_screen(self, current_scene: GameScene) -> None:
        """カード一覧画面に遷移する。

        Args:
            prev_scene (GameScene): 現在のシーン。
        """

        self._shared_vars.game_scene = GameScene.DISPLAY_CARDS
        self._shared_vars.prev_scene = current_scene
        self._screen_manager.transition.direction = "up"
        self._screen_manager.current = "cards_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _to_game_screen_from_card_list(self, prev_scene: GameScene) -> None:
        """カード一覧画面からゲーム画面に戻る。

        Args:
            prev_scene (GameScene): ひとつ前のシーン。
        """

        self._shared_vars.game_scene = prev_scene
        self._shared_vars.prev_scene = GameScene.DISPLAY_CARDS
        self._screen_manager.transition.direction = "down"
        self._screen_manager.current = "game_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _to_title_screen_from_game(self, crrent_scene: GameScene) -> None:
        """ゲーム中の画面からタイトル画面に遷移する。

        Args:
            prev_scene (GameScene): 現在のシーン。
        """

        self._shared_vars.game_scene = GameScene.TITLE
        self._shared_vars.prev_scene = crrent_scene
        self._screen_manager.transition.direction = "right"
        self._screen_manager.current = "title_screen"

    def _to_version_screen(self) -> None:
        """バージョン情報画面に遷移する。"""

        self._shared_vars.game_scene = GameScene.VERSION
        self._shared_vars.prev_scene = GameScene.MENU
        self._screen_manager.transition.direction = "up"
        self._screen_manager.current = "version_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _to_results_screen_from_menu(self) -> None:
        """メニュー画面から対戦成績画面に遷移する。"""

        self._shared_vars.game_scene = GameScene.DISPLAY_RESULTS
        self._shared_vars.prev_scene = GameScene.MENU
        self._screen_manager.transition.direction = "up"
        self._screen_manager.current = "results_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _to_next_screen(self) -> None:
        """次の画面に遷移する。"""

        if self._shared_vars.game_scene == GameScene.TITLE:
            self._display_menu_screen(GameScene.TITLE, "left")
        elif self._shared_vars.game_scene == GameScene.MENU:
            self._display_game_screen()
        elif self._shared_vars.game_scene == GameScene.DISPLAY_RESULTS:
            if self._shared_vars.prev_scene == GameScene.MENU:
                self._display_menu_screen(GameScene.DISPLAY_RESULTS, "down")
            else:
                self._display_title_screen()
        elif self._shared_vars.game_scene == GameScene.VERSION:
            self._display_menu_screen(GameScene.VERSION, "down")
        elif self._shared_vars.game_scene == GameScene.GAME_END:
            self._display_results_screen()

    def _display_menu_screen(
        self, current_scene: GameScene, transition_direction: str
    ) -> None:
        """メニュー画面を表示する。

        Args:
            prev_scene (GameScene): 現在のシーン。
            transition_direction (str): 画面遷移の方向。
        """

        self._shared_vars.game_scene = GameScene.MENU
        self._shared_vars.prev_scene = current_scene
        self._screen_manager.transition.direction = transition_direction
        self._screen_manager.current = "menu_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _display_game_screen(self) -> None:
        """ゲーム画面を表示する。"""

        self._shared_vars.game_scene = GameScene.EXCHANGE_CARDS
        self._shared_vars.prev_scene = GameScene.MENU
        self._screen_manager.transition.direction = "left"
        self._screen_manager.current = "game_screen"

        self._shared_vars.sounds[SoundScene.OPENING].stop()
        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()

    def _display_results_screen(self) -> None:
        """対戦成績画面を表示する。"""

        self._shared_vars.game_scene = GameScene.DISPLAY_RESULTS
        self._shared_vars.prev_scene = GameScene.GAME_END
        self._screen_manager.transition.direction = "left"
        self._screen_manager.current = "results_screen"

    def _display_title_screen(self) -> None:
        """タイトル画面を表示する。"""

        self._shared_vars.game_scene = GameScene.TITLE
        self._shared_vars.prev_scene = GameScene.DISPLAY_RESULTS
        self._screen_manager.transition.direction = "left"
        self._screen_manager.current = "title_screen"

        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()
