"""
対戦成績画面を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import copy
from typing import Callable, override

from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.uix.button import Button
from kivy.uix.recycleview import RecycleView
from kivy.uix.screenmanager import Screen

import constants
from models.my_config import MyConfig
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene
from views.record import Record  # KVコード内で使用
from views.yes_no_dialog import YesNoDialog

Builder.load_string("""
<ResultsScreen>:
    BoxLayout:
        orientation: 'vertical'
        canvas.before:
            Color:
                rgba: 144/256, 238/256, 144/256, 1.0
            Rectangle:
                pos: self.pos
                size: self.size

        AnchorLayout:
            anchor_y: 'bottom'
            size_hint_y: 0.2
            Image:
                size_hint: 0.6, 0.6
                source: "results.png"
        AnchorLayout:
            size_hint_y: 0.1
            anchor_x: 'right'
            anchor_y: 'top'
            padding: dp(36), 0
            Button:
                id: clear_button
                text: "対戦成績のクリア"
                font_size: dp(16)
                color: 0.0, 0.0, 0.0, 1.0
                size_hint: 0.24, 0.6
                background_normal: ''
                background_color: 0.8, 0.8, 0.8, 1.0
                canvas.before:
                    Color:
                        rgba: 0.0, 0.0, 0.0, 1.0
                    Line:
                        rectangle: [*self.pos, *self.size]
        AnchorLayout:
            size_hint_y: 0.55
            BoxLayout:
                orientation: 'vertical'
                size_hint: 0.9, 1.0
                BoxLayout:
                    size_hint_y: 1/10
                    HeaderLabel:
                        size_hint_x: 0.2
                        text: '人数／難易度'
                    HeaderLabel:
                        size_hint_x: 0.14
                        text: 'プレイ回数'
                    HeaderLabel:
                        size_hint_x: 0.22
                        text: '優勝'
                    HeaderLabel:
                        size_hint_x: 0.22
                        text: '引き分け'
                    HeaderLabel:
                        size_hint_x: 0.22
                        text: '失格'
                RecycleView:
                    id: rv
                    size_hint_y: 9/10
                    viewclass: 'Record'
                    canvas.before:
                        Color:
                            rgba: (64/255, 64/255, 64/255, 1.0)
                        Rectangle:
                            pos: self.pos
                            size: self.size
                    RecycleBoxLayout:
                        default_size: 0, rv.height/9
                        default_size_hint: 1, None
                        size_hint_y: None
                        height: self.minimum_height
                        orientation: 'vertical'
        AnchorLayout:
            size_hint_y: 0.15
            anchor_x: 'center'
            anchor_y: 'center'
            Label:
                text: '画面をクリックしてください'
                font_size: dp(20)
                color: 0.0, 0.4, 0.0, 1.0

<HeaderLabel@Label>:
    color: (0.0, 0.0, 0.0, 1.0)
    text_size: self.size
    valign: 'center'
    halign: 'center'
    canvas.before:
        Color:
            rgba: (180/255, 180/255, 180/255, 1.0)
        Rectangle:
            pos: self.pos
            size: self.size
        Color:
            rgba: (0.5, 0.5, 0.5, 1.0)
        Line:
            rectangle: (self.x + 1, self.y + 1, self.width, self.height)
            width: 1.0

<Record>:
    row: 'odd'
    mode: ''
    play_count: 0
    win_count: 0
    draw_count: 0
    dropout_count: 0

    orientation: 'horizontal'
    DataLabel:
        text: root.mode
        row: root.row
        size_hint_x: 0.2
        text_size: self.size
        halign: 'left'
    DataLabel:
        text: f'{root.play_count:,}'
        row: root.row
        size_hint_x: 0.14
        text_size: self.size
        halign: 'right'
    DataLabel:
        text: root.get_display_value(root.win_count, root.play_count)
        row: root.row
        size_hint_x: 0.22
        text_size: self.size
        halign: 'right'
    DataLabel:
        text: root.get_display_value(root.draw_count, root.play_count)
        row: root.row
        size_hint_x: 0.22
        text_size: self.size
        halign: 'right'
    DataLabel:
        text: root.get_display_value(root.dropout_count, root.play_count)
        row: root.row
        size_hint_x: 0.22
        text_size: self.size
        halign: 'right'

<DataLabel@Label>:
    row: 'odd'
    color: (0.0, 0.0, 0.0, 1.0)
    text_size: self.size
    valign: 'center'
    padding: dp(6), 0
    canvas.before:
        Color:
            rgba: (1.0, 1.0, 1.0, 1.0) if (root.row == 'odd') else (1.0, 0.85, 0.8, 1.0)
        Rectangle:
            pos: self.pos
            size: self.size
        Color:
            rgba: (0.5, 0.5, 0.5, 1.0)
        Line:
            rectangle: (self.x + 1, self.y + 1, self.width, self.height)
            width: 1.0
""")


class ResultsScreen(Screen):
    """対戦成績画面を管理するクラス。"""

    def __init__(
        self,
        config: MyConfig,
        shared_vars: SharedVars,
        to_next_screen: Callable[[], None],
        **kwargs,
    ) -> None:
        """対戦成績画面を管理するクラスのインスタンスを作成して返す。

        Args:
            config (MyConfig): アプリの設定値を管理するクラスのインスタンス。
            shared_vars (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
            to_next_screen (Callable[[], None]): 次の画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._config = config
        """アプリの設定値を管理するクラスのインスタンス"""

        self._shared_vars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._to_next_screen: Callable[[], None] = to_next_screen
        """次の画面に遷移するための関数"""

        self._clear_button: Button = self.ids.clear_button
        """対戦成績のクリアボタン"""

        self._rv: RecycleView = self.ids.rv
        """対戦成績表示用の RecycleView"""

    @override
    def on_pre_enter(self, *_) -> None:
        """画面が表示される前の処理。"""

        super().on_pre_enter()

        self._rv.data = self._config.results
        self._rv.refresh_from_data()

    @override
    def on_touch_down(self, touch: MotionEvent) -> None:
        """画面がタッチされたときの処理。

        Args:
            touch (MotionEvent): タッチイベントの情報。
        """

        super().on_touch_down(touch)

        if self._clear_button.collide_point(touch.x, touch.y):
            self._show_message_for_clear_results()
        else:
            self._to_next_screen()

    def _show_message_for_clear_results(self) -> None:
        """対戦成績のクリアの確認メッセージを表示する。"""

        msg: str = "対戦成績をクリアします。よろしいですか？"
        _ = YesNoDialog(message=msg, on_yes_callback=self._clear_results).show()

    def _clear_results(self) -> None:
        """対戦成績をクリアする。"""

        self._config.results = copy.deepcopy(constants.INITIAL_RESULTS)
        self._rv.data = self._config.results
        self._rv.refresh_from_data()

        self._shared_vars.sounds[SoundScene.RECOVERY].play()
