"""
メニュー画面を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from typing import Callable, override

from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.uix.button import Button
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen

from models.difficulty import Difficulty
from models.my_config import MyConfig
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene
from views.custom_image import CustomImage  # KVコード内で使用

Builder.load_string("""
<MenuScreen>:
    BoxLayout:
        orientation: 'vertical'
        canvas.before:
            Color:
                rgba: 144/256, 238/256, 144/256, 1.0
            Rectangle:
                pos: self.pos
                size: self.size

        AnchorLayout:
            size_hint_y: 0.12
            padding: 0, dp(16), 0, 0
            Image:
                size_hint: 0.9, 0.9
                source: "menu.png"
        AnchorLayout:
            size_hint_y: 0.06
            padding: 0, dp(16), 0, 0
            Label:
                text: "プレイ人数"
                font_size: dp(26)
                color: 0.0, 0.4, 0.0, 1.0
        AnchorLayout:
            size_hint_y: 0.12
            BoxLayout:
                size_hint_x: 0.6
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.2
                    CustomImage:
                        id: two_players_image
                        size_hint: None, None
                        size: dp(120 * 0.75), dp(80 * 0.75)
                        source: "2players_off.png"
                AnchorLayout:
                    size_hint_x: 0.2
                    CustomImage:
                        id: three_players_image
                        size_hint: None, None
                        size: dp(120 * 0.75), dp(80 * 0.75)
                        source: "3players_off.png"
                AnchorLayout:
                    size_hint_x: 0.2
                    CustomImage:
                        id: four_players_image
                        size_hint: None, None
                        size: dp(120 * 0.75), dp(80 * 0.75)
                        source: "4players_off.png"
                AnchorLayout:
                    size_hint_x: 0.2
                    CustomImage:
                        id: five_players_image
                        size_hint: None, None
                        size: dp(120 * 0.75), dp(80 * 0.75)
                        source: "5players_off.png"
                AnchorLayout:
                    size_hint_x: 0.2
                    CustomImage:
                        id: six_players_image
                        size_hint: None, None
                        size: dp(120 * 0.75), dp(80 * 0.75)
                        source: "6players_on.png"
        AnchorLayout:
            size_hint_y: 0.06
            padding: 0, dp(16), 0, 0
            Label:
                text: "難易度（２人プレイ以外）"
                font_size: dp(24)
                color: 0.0, 0.4, 0.0, 1.0
        AnchorLayout:
            size_hint_y: 0.12
            BoxLayout:
                size_hint_x: 0.5
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.5
                    CustomImage:
                        id: normal_mode_image
                        size_hint: None, None
                        size: dp(250 * 0.75), dp(80 * 0.75)
                        source: "normal_on.png"
                AnchorLayout:
                    size_hint_x: 0.5
                    CustomImage:
                        id: hard_mode_image
                        size_hint: None, None
                        size: dp(250 * 0.75), dp(80 * 0.75)
                        source: "hard_off.png"
        AnchorLayout:
            size_hint_y: 0.1
            BoxLayout:
                size_hint_x: 0.5
                orientation: 'horizontal'
                AnchorLayout:
                    size_hint_x: 0.05
                    Image:
                        id: card_addition_image
                        source: "check_on.png"
                Label:
                    id: card_addition_label
                    size_hint_x: 0.95
                    text: "「台風 (少)」カードを追加する"
                    font_size: dp(24)
                    color: 0.0, 0.0, 0.8, 1.0
        AnchorLayout:
            size_hint_y: 0.1
            BoxLayout:
                orientation: 'horizontal'
                size_hint_x: 0.4
                size_hint_y: 0.5
                spacing: dp(20)
                Button:
                    id: go_to_results_button
                    text: "対戦成績表示"
                    font_size: dp(16)
                    color: 0.0, 0.0, 0.0, 1.0
                    background_normal: ''
                    background_color: 0.8, 0.8, 0.8, 1.0
                    canvas.before:
                        Color:
                            rgba: 0.0, 0.0, 0.0, 1.0
                        Line:
                            rectangle: [*self.pos, *self.size]
                Button:
                    id: go_to_version_button
                    text: "バージョン情報"
                    font_size: dp(16)
                    color: 0.0, 0.0, 0.0, 1.0
                    background_normal: ''
                    background_color: 0.8, 0.8, 0.8, 1.0
                    canvas.before:
                        Color:
                            rgba: 0.0, 0.0, 0.0, 1.0
                        Line:
                            rectangle: [*self.pos, *self.size]
        AnchorLayout:
            size_hint_y: 0.12
            BoxLayout:
                orientation: 'horizontal'
                size_hint_x: 0.6
                size_hint_y: 0.6
                spacing: dp(4)
                Label:
                    size_hint_x: 0.68
                    text: "BGMの音量（0 ～ 100）"
                    font_size: dp(28)
                    color: 1.0, 0.5, 0.0, 1.0
                Button:
                    id: volume_decrease_button
                    size_hint_x: 0.08
                    text: "－"
                    font_size: dp(28)
                    background_normal: ''
                    background_color: 1.0, 0.4, 0.0, 1.0
                Label:
                    id: volume_label
                    size_hint_x: 0.16
                    padding: dp(8), 0
                    text_size: self.size
                    halign: 'right'
                    valign: 'middle'
                    font_size: dp(26)
                    color: 0.0, 0.0, 0.0, 1.0
                    canvas.before:
                        Color:
                            rgba: 1.0, 1.0, 1.0, 1.0
                        Rectangle:
                            pos: self.pos
                            size: self.size
                Button:
                    id: volume_increase_button
                    size_hint_x: 0.08
                    text: "＋"
                    font_size: dp(28)
                    background_normal: ''
                    background_color: 1.0, 0.4, 0.0, 1.0
        AnchorLayout:
            size_hint_y: 0.18
            Button:
                id: start_button
                text: "ゲーム開始"
                font_size: dp(20)
                color: 0.0, 0.0, 0.0, 1.0
                size_hint: 0.25, 0.5
                background_normal: ''
                background_color: 0.8, 0.8, 0.8, 1.0
                canvas.before:
                    Color:
                        rgba: 0.0, 0.0, 0.0, 1.0
                    Line:
                        rectangle: [*self.pos, *self.size]
""")


class MenuScreen(Screen):
    """メニュー画面を管理するクラス。"""

    def __init__(
        self,
        config: MyConfig,
        shared_vars: SharedVars,
        to_next_screen: Callable[[], None],
        to_results_screen: Callable[[], None],
        to_version_screen: Callable[[], None],
        **kwargs,
    ) -> None:
        """メニュー画面を管理するクラスのインスタンスを作成して返す。

        Args:
            config (MyConfig): アプリの設定値を管理するクラスのインスタンス。
            shared_vars (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
            to_next_screen (Callable[[], None]): 次の画面に遷移するための関数。
            to_results_screen (Callable[[], None]): 対戦成績画面に遷移するための関数。
            to_version_screen (Callable[[], None]): バージョン情報画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._config = config
        """アプリの設定値を管理するクラスのインスタンス"""

        self._shared_vars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._to_next_screen: Callable[[], None] = to_next_screen
        """次の画面に遷移するための関数"""

        self._to_results_screen: Callable[[], None] = to_results_screen
        """対戦成績画面に遷移するための関数"""

        self._to_version_screen: Callable[[], None] = to_version_screen
        """バージョン情報画面に遷移するための関数"""

        self._start_button: Button = self.ids.start_button
        self._go_to_results_button: Button = self.ids.go_to_results_button
        self._go_to_version_button: Button = self.ids.go_to_version_button
        self._2players_image: Image = self.ids.two_players_image
        self._3players_image: Image = self.ids.three_players_image
        self._4players_image: Image = self.ids.four_players_image
        self._5players_image: Image = self.ids.five_players_image
        self._6players_image: Image = self.ids.six_players_image
        self._normal_mode_image: Image = self.ids.normal_mode_image
        self._hard_mode_image: Image = self.ids.hard_mode_image
        self._card_addition_image: Image = self.ids.card_addition_image
        self._card_addition_label: Label = self.ids.card_addition_label
        self._volume_label: Label = self.ids.volume_label
        self._volume_decrease_button: Button = self.ids.volume_decrease_button
        self._volume_increase_button: Button = self.ids.volume_increase_button

    @override
    def on_pre_enter(self, *_) -> None:
        """画面が表示される前の処理。"""

        super().on_pre_enter()

        self._update_display(first=True)

    @override
    def on_touch_up(self, touch: MotionEvent) -> None:
        """画面がタッチされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        super().on_touch_up(touch)

        if self._2players_image.collide_point(*touch.pos):
            self._config.player_count = 2
            self._update_display()
        elif self._3players_image.collide_point(*touch.pos):
            self._config.player_count = 3
            self._update_display()
        elif self._4players_image.collide_point(*touch.pos):
            self._config.player_count = 4
            self._update_display()
        elif self._5players_image.collide_point(*touch.pos):
            self._config.player_count = 5
            self._update_display()
        elif self._6players_image.collide_point(*touch.pos):
            self._config.player_count = 6
            self._update_display()
        elif self._normal_mode_image.collide_point(*touch.pos):
            self._config.difficulty = Difficulty.NORMAL
            self._update_display()
        elif self._hard_mode_image.collide_point(*touch.pos):
            self._config.difficulty = Difficulty.HARD
            self._update_display()
        elif self._card_addition_image.collide_point(*touch.pos):
            self._config.add_option_cards = not self._config.add_option_cards
            self._update_display()
        elif self._card_addition_label.collide_point(*touch.pos):
            self._config.add_option_cards = not self._config.add_option_cards
            self._update_display()
        elif self._volume_increase_button.collide_point(*touch.pos):
            self._volume_increase()
        elif self._volume_decrease_button.collide_point(*touch.pos):
            self._volume_decrease()
        elif self._go_to_results_button.collide_point(*touch.pos):
            self._to_results_screen()
        elif self._go_to_version_button.collide_point(*touch.pos):
            self._to_version_screen()
        elif self._start_button.collide_point(*touch.pos):
            self._to_next_screen()
        else:
            pass

    def _volume_increase(self) -> None:
        """音量を上げる。"""

        if self._config.bgm_volume < 100:
            self._config.bgm_volume += 1
            self._change_volume()

    def _volume_decrease(self) -> None:
        """音量を下げる。"""

        if self._config.bgm_volume > 0:
            self._config.bgm_volume -= 1
            self._change_volume()

    def _change_volume(self) -> None:
        """音量を変更する。"""

        for scene in SoundScene:
            self._shared_vars.sounds[scene].volume = self._get_volume(scene)

        self._shared_vars.sounds[SoundScene.CLICK].play()
        self._volume_label.text = str(self._config.bgm_volume)

    def _get_volume(self, scene: SoundScene) -> float:
        """個別に調整された音量を返す。

        Args:
            scene (SoundScene): BGM / 効果音の再生シーン。

        Returns:
            float: 個別に調整された音量。
        """

        volume: float = self._config.bgm_volume * scene.volume_scale
        volume = 100.0 if volume > 100.0 else volume

        return volume / 100.0

    def _update_display(self, first: bool = False) -> None:
        """画面の表示を更新する。

        Args:
            first (bool, optional): 初回表示の場合は True。
        """

        images: list[Image] = [
            self._2players_image,
            self._3players_image,
            self._4players_image,
            self._5players_image,
            self._6players_image,
        ]
        for i, image in enumerate(images, start=2):
            if i == self._config.player_count:
                image.source = f"{i}players_on.png"
            else:
                image.source = f"{i}players_off.png"

        if self._config.difficulty == Difficulty.NORMAL:
            self._normal_mode_image.source = "normal_on.png"
            self._hard_mode_image.source = "hard_off.png"
        else:
            self._normal_mode_image.source = "normal_off.png"
            self._hard_mode_image.source = "hard_on.png"

        if first:
            self._volume_label.text = str(self._config.bgm_volume)

        if self._config.add_option_cards:
            self._card_addition_image.source = "check_on.png"
        else:
            self._card_addition_image.source = "check_off.png"

        if not first:
            self._shared_vars.sounds[SoundScene.CLICK].play()
