"""
ゲーム中の画面を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import time
import webbrowser
from typing import Callable, override

from kivy.core.window import Window
from kivy.graphics import Color, Line, Rectangle
from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.metrics import dp
from kivy.uix.button import Button
from kivy.uix.floatlayout import FloatLayout
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen
from kivy.uix.textinput import TextInput
from kivy.uix.widget import Widget

import constants
from controllers.game_manager import GameManager
from models.card import Card
from models.difficulty import Difficulty
from models.game_scene import GameScene
from models.my_config import MyConfig
from models.player import Player
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene
from views.custom_image import CustomImage
from views.entering_release_dialog import EnteringReleaseDialog
from views.yes_no_dialog import YesNoDialog

Builder.load_string("""
<GameScreen>:
    FloatLayout:
        size: root.size
        pos: root.pos
        canvas:
            Rectangle:
                source: "game_screen_bg.png"
                pos: self.pos
                size: self.size
        TextInput:
            id: game_status_text
            pos_hint: {'x': 0.03, 'y': 0.69}
            size_hint: 0.60, 0.26
            readonly: True
            background_normal: ''
            background_color: 1.0, 1.0, 1.0, 1.0
            background_active: ''
            padding: dp(12), dp(8)
            line_spacing: dp(4)
            canvas.before:
                Color:
                    rgba: 0.0, 0.0, 0.0, 1.0
                Line:
                    rectangle: [*self.pos, *self.size]
        Label:
            id: game_mode_label
            pos_hint: {'x': 0.68, 'y': 0.875}
            size_hint: 0.29, 0.08
            font_size: dp(24)
            color: 1.0, 1.0, 1.0, 1.0
            text_size: self.size
            halign: 'center'
            valign: 'middle'
            canvas.before:
                Color:
                    rgba: 160/256, 44/256, 44/256, 1.0
                Rectangle:
                    pos: self.pos
                    size: self.size
        Label:
            id: turn_order_label
            pos_hint: {'x': 0.68, 'y': 0.795}
            size_hint: 0.29, 0.06
            font_size: dp(20)
            color: 1.0, 1.0, 1.0, 1.0
            text_size: self.size
            halign: 'center'
            valign: 'middle'
        Label:
            id: card_description_label
            pos_hint: {'x': 0.4, 'y': 0.263}
            size_hint: 0.57, 0.07
            font_size: dp(14)
            color: 0.0, 0.0, 0.0, 1.0
            line_height: 1.2
            text_size: self.size
            halign: 'left'
            valign: 'middle'
        Button:
            id: card_list_button
            text: "カード一覧表示"
            color: 0.0, 0.0, 0.0, 1.0
            font_size: dp(16)
            pos_hint: {'center_x': 0.5, 'center_y': 0.56}
            size_hint: 0.2, 0.06
            background_normal: ''
            background_color: 0.8, 0.8, 0.8, 1.0
            canvas.before:
                Color:
                    rgba: 0.0, 0.0, 0.0, 1.0
                Line:
                    rectangle: [*self.pos, *self.size]
        Button:
            id: exit_game_button
            text: "ゲームを中断"
            color: 0.0, 0.0, 0.0, 1.0
            font_size: dp(16)
            pos_hint: {'center_x': 0.5, 'center_y': 0.63}
            size_hint: 0.2, 0.06
            background_normal: ''
            background_color: 0.8, 0.8, 0.8, 1.0
            canvas.before:
                Color:
                    rgba: 0.0, 0.0, 0.0, 1.0
                Line:
                    rectangle: [*self.pos, *self.size]
        Label:
            id: bgm_info_label
            markup: True
            font_size: dp(16)
            color: 0.6, 0.3, 0.0, 1.0
            pos_hint: {'center_x': 0.5, 'center_y': 0.042}
            size_hint: 0.8, 0.05

    # カード等を配置するためのレイアウト
    FloatLayout:
        id: game_board
        size: root.size
        pos: root.pos
""")


class GameScreen(Screen):
    """ゲーム中の画面を管理するクラス。"""

    def __init__(
        self,
        config: MyConfig,
        shared_vars: SharedVars,
        logger: logging.Logger,
        to_next_screen: Callable[[], None],
        to_cards_screen: Callable[[GameScene], None],
        to_title_screen: Callable[[GameScene], None],
        **kwargs,
    ) -> None:
        """ゲーム中の画面を管理するクラスのインスタンスを作成して返す。

        Args:
            config (MyConfig): アプリの設定値を管理するクラスのインスタンス。
            shared_vars (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
            logger (logging.Logger): logging.Logger のインスタンス。
            to_next_screen (Callable[[], None]): 次の画面に遷移するための関数。
            to_cards_screen (Callable[[GameScene], None]): カード一覧画面に遷移するための関数。
            to_title_screen (Callable[[GameScene], None]): タイトル画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._config = config
        """アプリの設定値を管理するクラスのインスタンス"""

        self._shared_vars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

        self._gm: GameManager = GameManager()
        """ゲームの進行を管理するクラスのインスタンス"""

        self._to_next_screen: Callable[[], None] = to_next_screen
        """次の画面に遷移するための関数"""

        self._to_cards_screen: Callable[[GameScene], None] = to_cards_screen
        """カード一覧画面に遷移するための関数"""

        self._to_title_screen: Callable[[GameScene], None] = to_title_screen
        """タイトル画面に遷移するための関数"""

        self._game_board: FloatLayout = self.ids.game_board
        """カード等を表示するためのレイアウト"""
        self._game_status_text: TextInput = self.ids.game_status_text
        """ゲーム状況を表示するための TextInput"""
        self._game_mode_label: Label = self.ids.game_mode_label
        """ゲームモードを表示するためのラベル"""
        self._turn_order_label: Label = self.ids.turn_order_label
        """カードを出す順番を表示するためのラベル"""
        self._card_description_label: Label = self.ids.card_description_label
        """カードの説明を表示するためのラベル"""
        self._exit_game_button: Button = self.ids.exit_game_button
        """「ゲーム中断」ボタン"""
        self._card_list_button: Button = self.ids.card_list_button
        """「カード一覧表示」ボタン"""
        self._bgm_info_label: Label = self.ids.bgm_info_label
        """BGM の情報を表示するためのラベル"""

        self._bgm_info_label.bind(on_ref_press=self._open_link)  # type: ignore

    def _open_link(self, _, value: str) -> None:
        """URL のリンクを開く。

        Args:
            value (str): URL。
        """

        webbrowser.open(value)

    @override
    def on_enter(self, *_) -> None:
        """画面が表示されたときの処理。"""

        super().on_enter()

        Window.bind(mouse_pos=self._on_mouse_pos)
        self._game_mode_label.text = self._config.difficulty.get_mode_name(
            self._config.player_count
        )

        if self._shared_vars.prev_scene is GameScene.MENU:
            self._play_bgm()
            self._log_start_setting()
            self._gm = GameManager(
                self._config,
                self._shared_vars,
                self._logger,
                self._add_status,
            )

            text: str = "最初に、いらない手札を交換できます。\n"
            text += '交換したい手札を "クリック" して選択してください。\n'
            text += "選択した手札をもう一度クリックすると、選択を解除できます。\n"
            text += '選択が完了したら、画面上で "右クリック" してください。\n'
            self._add_status(text, is_new=True)

            self._update_display()
        else:
            # カード一覧画面から遷移してきた場合は無視
            pass

    def _play_bgm(self) -> None:
        """BGM を再生する。"""

        if self._config.difficulty is Difficulty.NORMAL:
            scene: SoundScene = SoundScene.NORMAL_MODE
        else:
            scene: SoundScene = SoundScene.HARD_MODE

        self._shared_vars.sounds[scene].play()
        self._display_bgm_description(scene)

    def _display_bgm_description(self, scene: SoundScene) -> None:
        """BGM の情報を表示する。

        Args:
            scene (SoundScene): 音声の再生シーン。
        """

        title: str = scene.title
        site_name: str = scene.site_name
        url: str = scene.site_url
        link: str = f"[color=3000ff][u][ref={url}]{url}[/ref][/u][/color]"
        self._bgm_info_label.text = f"音楽: 『{title}』（{site_name}: {link}）"

    def _log_start_setting(self) -> None:
        """ゲーム開始時の設定値をログに出力する。"""

        self._logger.info("-------- ゲームを開始しました --------")

        mode: str = self._config.difficulty.get_mode_name(self._config.player_count)
        self._logger.info(f"ゲームモード： {mode}")

        msg: str = "あり" if self._config.add_option_cards else "なし"
        self._logger.info(f"「台風（少）」カードの追加： {msg}")

    def _on_mouse_pos(self, _, mouse_pos) -> None:
        """マウスポインタの位置が変化したときの処理。

        Args:
            _ (Window): Window オブジェクト。
            mouse_pos (tuple[float, float]): マウスポインタの位置。
        """

        text: str = ""
        for card in self._gm.human_player.cards_in_hand:
            if card.widget.collide_point(*card.widget.parent.to_widget(*mouse_pos)):
                text = f"【{card.name}】\n　 {card.description}"
                break

        if any(self._gm.cards_on_table):
            card: Card = self._gm.cards_on_table[-1]
            if card.widget.collide_point(*card.widget.parent.to_widget(*mouse_pos)):
                text = f"【{card.name}】\n　 {card.description}"

        self._card_description_label.text = text

    @override
    def on_touch_down(self, touch: MotionEvent) -> None:
        """画面がタッチされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        def not_scroll() -> bool:
            """スクロール操作でない場合は True を返す。"""

            return (touch.button != "scrollup") and (touch.button != "scrolldown")

        super().on_touch_down(touch)

        game_scene: GameScene = self._shared_vars.game_scene
        if self._bgm_info_label.collide_point(*touch.pos):
            pass
        elif self._card_list_button.collide_point(*touch.pos) and not_scroll():
            Window.unbind(mouse_pos=self._on_mouse_pos)
            self._to_cards_screen(self._shared_vars.game_scene)
        elif self._exit_game_button.collide_point(*touch.pos) and not_scroll():
            self._show_message_for_abort()
        elif game_scene is GameScene.EXCHANGE_CARDS:
            if touch.button == "left":
                self._on_hand_touch(touch)
            elif touch.button == "right":
                self._show_message_for_exchange_hand()
        elif (game_scene is GameScene.TURN_OF_COMPUTER) and not_scroll():
            if any(self._gm.get_players_exceeding_range()):
                self._play_card(None)
            else:
                card: Card = self._gm.select_computer_card()
                self._play_card(card)
        elif (game_scene is GameScene.TURN_OF_HUMAN) and not_scroll():
            if any(self._gm.get_players_exceeding_range()):
                self._play_card(None)
            elif touch.is_double_tap:
                self._on_hand_touch(touch)
        elif (game_scene is GameScene.GAME_END) and not_scroll():
            self._game_status_text.text = ""
            Window.unbind(mouse_pos=self._on_mouse_pos)
            self._to_next_screen()
        else:
            pass

    def _show_message_for_abort(self) -> None:
        """ゲーム中断の確認メッセージを表示する。"""

        text: str = "ゲームを中断してタイトル画面に戻ります。\nよろしいですか？"
        _ = YesNoDialog(
            message=text, on_yes_callback=self._transition_to_title_scene
        ).show()

    def _transition_to_title_scene(self) -> None:
        """タイトル画面に遷移する。"""

        self._game_status_text.text = ""

        Window.unbind(mouse_pos=self._on_mouse_pos)

        self._logger.info("-------- ゲームを中断しました --------")

        self._stop_bgm()
        self._shared_vars.sounds[SoundScene.SCREEN_TRANSITION].play()
        time.sleep(0.3)

        self._to_title_screen(self._shared_vars.game_scene)

    def _stop_bgm(self) -> None:
        """BGM を停止する。"""

        for scene in [SoundScene.NORMAL_MODE, SoundScene.HARD_MODE]:
            if self._shared_vars.sounds[scene].state == "play":
                self._shared_vars.sounds[scene].stop()

    def _on_hand_touch(self, touch: MotionEvent) -> None:
        """人間プレイヤーの手札がタッチされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        hand: list[Card] = self._gm.human_player.cards_in_hand
        widgets_of_hand: list[Widget] = [c.widget for c in hand]
        for widget in widgets_of_hand:
            if widget.collide_point(*touch.pos):
                i: int = widgets_of_hand.index(widget)
                prev_widget: Widget = widgets_of_hand[i - 1]
                if (i > 0) and prev_widget.collide_point(*touch.pos):
                    pass
                else:
                    game_scene: GameScene = self._shared_vars.game_scene
                    if game_scene is GameScene.EXCHANGE_CARDS:
                        hand[i].is_selected = not hand[i].is_selected
                        self._shared_vars.sounds[SoundScene.CLICK].play()
                        self._update_display()
                    elif game_scene is GameScene.TURN_OF_HUMAN:
                        self._play_card(hand[i])
                    else:
                        pass

    def _show_message_for_exchange_hand(self) -> None:
        """手札の交換の確認メッセージを表示する。"""

        self._shared_vars.sounds[SoundScene.RECOVERY].play()

        select_count: int = len(
            [c for c in self._gm.human_player.cards_in_hand if c.is_selected]
        )
        if select_count > 0:
            text: str = f"手札を {select_count} 枚交換します。よろしいですか？"
        else:
            text: str = "手札は交換しません。よろしいですか？"

        _ = YesNoDialog(message=text, on_yes_callback=self._exchange_hand).show()

    def _exchange_hand(self) -> None:
        """すべてのプレイヤーの手札を交換する。"""

        for player in self._gm.players:
            exchanged_count: int = self._gm.exchange_hand(player)

            player.cards_in_hand.sort(key=lambda c: c.index)
            text: str = f"{player.name} は手札を"
            self._add_status(f"{text} {exchanged_count} 枚交換しました。")
            self._gm.log_starting_hand(
                player, is_before=False, exchanged_count=exchanged_count
            )

        self._play_se_for_exchange_hand()
        self._gm.return_played_cards_to_deck(return_all=True)
        self._gm.log_players_status()
        self._gm.log_card_count()
        self._gm.move_to_next_player(is_first=True)
        self._display_message_requesting_action()
        self._update_display()

    def _play_se_for_exchange_hand(self) -> None:
        """手札交換時の SE を再生する。

        Args:
            exchanged_count (int): 交換した手札の枚数。
        """

        for _ in range(2):
            self._shared_vars.sounds[SoundScene.CARD_FLIP].play()
            time.sleep(0.1)

    def _play_card(self, card: Card | None) -> None:
        """プレイヤーがカードを場に出す一連の処理。

        Args:
            card (Card | None): 出すカード。貯水量が範囲を超えたプレイヤーがいる場合は None。
        """

        if card:
            self._gm.play_card(card, self._transition_to_entering_release_dialog)
            if self._shared_vars.game_scene is GameScene.ENTERING_RELEASE:
                return
        else:
            players: list[Player] = self._gm.get_players_exceeding_range()
            recover_player: Player | None = self._gm.process_for_player_exceeding_range(
                players
            )

            if recover_player:
                self._gm.current_player = recover_player

        if self._gm.has_players_without_hand():
            self._gm.process_drop_out_players_after_game_end()
            self._finish_game()
        elif self._gm.is_game_end_by_drop_out():
            self._finish_game()
        elif (
            not any(self._gm.get_players_exceeding_range())
        ) and self._shared_vars.game_scene != GameScene.GAME_END:
            self._after_play_card()

        self._display_message_requesting_action()
        self._update_display()

    def _after_play_card(self) -> None:
        """カードを場に出した後の処理。"""

        self._gm.move_to_next_player()
        self._gm.return_played_cards_to_deck(return_all=False)
        self._gm.log_players_status()
        self._gm.log_card_count()

    def _transition_to_entering_release_dialog(self, initial_value: int) -> None:
        """放流量の入力ダイアログボックスを表示する。

        Args:
            initial_value (int): 放流後の貯水量の初期設定値。
        """

        self._dialog: EnteringReleaseDialog = EnteringReleaseDialog(
            current_amount=self._gm.current_player.water_storage,
            atrer_release=initial_value,
            sound_click=self._shared_vars.sounds[SoundScene.CLICK],
            callback=self._entering_release_dialog_closed,
        )
        """放流量の入力ダイアログボックスのインスタンス"""

        self._dialog.show()

        self._shared_vars.prev_scene = self._shared_vars.game_scene
        self._shared_vars.game_scene = GameScene.ENTERING_RELEASE

    def _entering_release_dialog_closed(self) -> None:
        """放流量の入力ダイアログボックスが閉じられたときの処理。"""

        recover_player: Player | None = None

        self._shared_vars.game_scene = self._shared_vars.prev_scene
        self._shared_vars.prev_scene = GameScene.ENTERING_RELEASE
        self._gm.apply_release_effect_for_human_player(self._dialog.after_release)

        if self._gm.has_players_without_hand():
            self._gm.process_drop_out_players_after_game_end()
            self._finish_game()
        elif any(players := self._gm.get_players_exceeding_range()):
            recover_player = self._gm.process_for_player_exceeding_range(players)

        if self._gm.is_game_end_by_drop_out():
            self._finish_game()
        elif (
            not any(self._gm.get_players_exceeding_range())
        ) and self._shared_vars.game_scene != GameScene.GAME_END:
            if recover_player:
                self._gm.current_player = recover_player

            self._after_play_card()

        self._display_message_requesting_action()
        self._update_display()

    def _finish_game(self) -> None:
        """ゲームを終了する。"""

        self._shared_vars.prev_scene = self._shared_vars.game_scene
        self._shared_vars.game_scene = GameScene.GAME_END
        self._gm.log_players_status(is_result=True)
        self._gm.output_winner()
        self._gm.record_result()

        time.sleep(0.3)
        self._play_exit_bgm()

        self._logger.info("-------- ゲームを終了しました --------")

    def _play_exit_bgm(self) -> None:
        """ゲーム終了時の BGM を再生する。"""

        self._stop_bgm()

        if self._gm.human_player.is_win:
            scene: SoundScene = SoundScene.WINNING_EXIT
        elif self._gm.human_player.is_drop_out:
            scene: SoundScene = SoundScene.DROP_OUT_EXIT
        elif self._gm.human_player.is_draw:
            scene: SoundScene = SoundScene.DRAW_EXIT
        else:
            scene: SoundScene = SoundScene.LOSE_EXIT

        self._shared_vars.sounds[scene].play()
        self._display_bgm_description(scene)

    ###############################################################################
    # ゲーム画面の表示の更新

    def _update_display(self) -> None:
        """ゲーム画面の表示を更新する。"""

        self._game_board.clear_widgets()

        for player in self._gm.players:
            player.cards_in_hand.sort(key=lambda c: c.index)
            self._game_board.add_widget(player.status)
            mark_text: str = self._get_player_mark(player)
            rank_text: str = self._get_player_rank(player)
            player.update_status(mark_text, rank_text)
            hand_count: int = len(player.cards_in_hand)

            if player.is_human:
                self._display_human_hand(player, hand_count)
            else:
                self._display_computer_hand(player, hand_count)

        self._display_card_on_table()
        self._display_turn_order()

    def _get_player_mark(self, player: Player) -> str:
        """プレイヤー名の先頭に表示するマークの文字列を返す。

        Args:
            player (Player): 対象となるプレイヤー。
        """

        scene_to_exclude: list[GameScene] = [
            GameScene.EXCHANGE_CARDS,
            GameScene.GAME_END,
        ]

        if player.is_drop_out:
            return "失格"
        elif self._shared_vars.game_scene not in scene_to_exclude:
            return "順番" if (self._gm.current_player is player) else ""
        elif self._shared_vars.game_scene is GameScene.GAME_END:
            return "優勝" if player.is_win else ""
        else:
            return ""

    def _get_player_rank(self, player: Player) -> str:
        """ゲーム中のプレイヤーの貯水量の順位を返す。

        Args:
            player (Player): 対象となるプレイヤー。
        """

        scene_to_exclude: list[GameScene] = [
            GameScene.EXCHANGE_CARDS,
            GameScene.GAME_END,
        ]

        if self._shared_vars.game_scene not in scene_to_exclude:
            active_players: list[Player] = [
                p for p in self._gm.players if not p.is_drop_out
            ]
            largest: int = max([p.water_storage for p in active_players])
            second_players: list[Player] = [
                p for p in active_players if (p.water_storage < largest)
            ]
            if any(second_players):
                second_largest: int = max([p.water_storage for p in second_players])
            else:
                second_largest: int = 0

            water_storage: int = player.water_storage
            capacity: int = constants.WATER_CAPACITY
            if (not player.is_drop_out) and (water_storage > capacity):
                return "越水"
            elif (not player.is_drop_out) and (water_storage <= 0):
                return "渇水"
            else:
                if second_largest == 0:
                    return ""

                if (not player.is_drop_out) and (water_storage == largest):
                    return "最多"
                elif (not player.is_drop_out) and (water_storage == second_largest):
                    return "２位"
                else:
                    return ""
        else:
            return ""

    def _display_human_hand(self, player: Player, hand_count: int) -> None:
        """人間プレイヤーの手札を表示する。

        Args:
            player (Player): 対象となるプレイヤー。
            hand_count (int): 手札の枚数。
        """

        for i, card in enumerate(player.cards_in_hand):
            width_hint: float = self._get_width_hint(
                hand_count, 12, dp(60.0), dp(2.0), dp(6.0)
            )
            x: float = player.base_point["x"] + width_hint * i + 0.019
            y: float = player.base_point["y"] - 0.174
            card.widget.pos_hint = {"x": x, "y": y}
            self._game_board.add_widget(card.widget, index=i)

            if (
                self._shared_vars.game_scene == GameScene.EXCHANGE_CARDS
                and card.is_selected
            ):
                path: str = "check_mark.png"
                image: CustomImage = CustomImage(
                    source=path, size=(dp(60), dp(90)), pos_hint={"x": x, "y": y}
                )
                self._game_board.add_widget(image)

    def _display_computer_hand(self, player: Player, hand_count: int) -> None:
        """コンピュータプレイヤーの手札（裏側）を表示する。

        Args:
            player (Player): 対象となるプレイヤー。
            hand_count (int): 手札の枚数。
        """

        for i in range(hand_count):
            width_hint: float = self._get_width_hint(
                hand_count, 9, dp(24.0), dp(2.0), dp(2.0)
            )
            x: float = player.base_point["x"] + width_hint * i + 0.014
            y: float = player.base_point["y"] - 0.075
            path: str = "back_of_card_s.png"
            image: CustomImage = CustomImage(source=path, size=(dp(24), dp(36)))
            image.pos_hint = {"x": x, "y": y}
            self._game_board.add_widget(image, index=i)

    def _get_width_hint(
        self,
        hand_count: int,
        limit: int,
        width: float,
        space: float,
        right_margin: float,
    ) -> float:
        """画面の横幅に対するカードの相対幅を算出して返す。

        Args:
            hand_count (int): 手札の枚数。
            limit (int): 並べる手札の枚数の上限。
            width (float): カードの幅。
            space (float): カードの間隔。
            right_margine (float): 右端の余白。

        Returns:
            float: カードの相対幅。
        """

        if hand_count <= limit:
            width_hint: float = (width + space) / self.width
        else:
            total_width: float = (width + space) * limit - right_margin
            width_hint: float = total_width / hand_count / self.width

        return width_hint

    def _display_card_on_table(self) -> None:
        """場に出された最後のカードを表示する。"""

        if any(self._gm.cards_on_table):
            image: CustomImage = self._gm.cards_on_table[-1].widget
            image.pos_hint = {"center_x": 0.5, "center_y": 0.432}
            self._game_board.add_widget(image)

    def _display_turn_order(self) -> None:
        """カードを出す順番を表示する。"""

        if self._gm.is_clockwise:
            self._turn_order_label.text = "順番： 時計回り"
            self._turn_order_label.color = (1.0, 1.0, 1.0, 1.0)
            rect_color: Color = Color(rgba=(200 / 256, 55 / 256, 55 / 256, 1.0))
        else:
            self._turn_order_label.text = "順番： 反時計回り"
            self._turn_order_label.color = (200 / 256, 55 / 256, 55 / 256, 1.0)
            rect_color: Color = Color(rgba=(1.0, 1.0, 1.0, 1.0))

        rect: Rectangle = Rectangle(
            pos=self._turn_order_label.pos, size=self._turn_order_label.size
        )
        border_color: Color = Color(rgba=(200 / 256, 55 / 256, 55 / 256, 1.0))
        line: Line = Line(
            rectangle=(*self._turn_order_label.pos, *self._turn_order_label.size),
            width=dp(1),
        )

        self._turn_order_label.canvas.before.clear()  # type: ignore
        self._turn_order_label.canvas.before.add(rect_color)  # type: ignore
        self._turn_order_label.canvas.before.add(rect)  # type: ignore
        self._turn_order_label.canvas.before.add(border_color)  # type: ignore
        self._turn_order_label.canvas.before.add(line)  # type: ignore

    ###############################################################################
    # ゲーム状況の表示

    def _display_message_requesting_action(self) -> None:
        """プレイヤーに操作を要求するメッセージを表示する。"""

        if self._shared_vars.game_scene is GameScene.GAME_END:
            self._add_status("")
            self._add_status("ゲーム終了です。")
            self._add_status('▶ 画面を "クリック" してください。\n')
        elif any(self._gm.get_players_exceeding_range()):
            self._add_status("")
            self._add_status('▶ 画面を "クリック" してください。\n')
        elif self._shared_vars.game_scene is GameScene.TURN_OF_HUMAN:
            self._add_status("")
            self._add_status("あなた の番です。")
            self._add_status('▶ 出したい手札を "ダブルクリック" してください。\n')
        elif self._shared_vars.game_scene is GameScene.TURN_OF_COMPUTER:
            self._add_status("")
            self._add_status(f"{self._gm.current_player.name} の番です。")
            self._add_status('▶ 画面を "クリック" してください。\n')

    def _add_status(self, line: str, is_new: bool = False) -> None:
        """ゲーム状況の表示に 1 行分のテキストを追加する。

        Args:
            line (str): 表示するテキスト 1 行分。
            is_new (bool, optional): 表示をクリアする場合は True。
        """

        if is_new:
            self._game_status_text.text = line
        else:
            self._game_status_text.text += f"\n{line}"
