"""
放流量の入力ダイアログボックスの定義。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from typing import Callable

from kivy.core.audio import Sound
from kivy.core.window import Keyboard, Window
from kivy.lang import Builder
from kivy.properties import NumericProperty, ObjectProperty
from kivy.uix.anchorlayout import AnchorLayout
from kivy.uix.button import Button
from kivy.uix.label import Label
from kivy.uix.modalview import ModalView

Builder.load_string("""
<EnteringReleaseDialogLayout>:
    size_hint: 0.8, 0.8
    BoxLayout:
        orientation: 'vertical'
        Label:
            size_hint_y: 0.2
            font_size: dp(18)
            text_size: self.size
            halign: 'center'
            valign: 'middle'
            text: "「放流後の貯水量」を決定してください"
        Label:
            size_hint_y: 0.2
            font_size: dp(18)
            color: 0.7, 1.0, 0.7, 1.0
            text_size: self.size
            halign: 'center'
            valign: 'middle'
            text: f"現在の貯水量： {int(root.current_amount * 100):,} 万㎥"
        AnchorLayout:
            size_hint_y: 0.3
            anchor_x: 'center'
            anchor_y: 'center'
            BoxLayout:
                orientation: 'horizontal'
                size_hint: 1.0, 0.6
                spacing: dp(4)
                Label:
                    size_hint_x: 0.45
                    text: "放流後の貯水量："
                    font_size: dp(20)
                    color: 1.0, 0.6, 0.6, 1.0
                    text_size: self.size
                    halign: 'center'
                    valign: 'middle'
                Button:
                    id: decrement_button
                    size_hint_x: 0.1
                    text: "－"
                    font_size: dp(24)
                    on_release: root.on_decrement_button_click()
                Label:
                    id: post_release_volume_label
                    size_hint_x: 0.35
                    font_size: dp(18)
                    color: 0.0, 0.0, 0.0, 1.0
                    text_size: self.size
                    halign: 'right'
                    valign: 'middle'
                    padding: dp(8), 0
                    canvas.before:
                        Color:
                            rgba: 1.0, 1.0, 1.0, 1.0
                        Rectangle:
                            pos: self.pos
                            size: self.size
                Button:
                    id: increment_button
                    size_hint_x: 0.1
                    text: "＋"
                    font_size: dp(24)
                    on_release: root.on_increment_button_click()
        AnchorLayout:
            size_hint_y: 0.3
            anchor_x: 'center'
            anchor_y: 'bottom'
            Button:
                size_hint: 0.4, 0.7
                text: "決定"
                font_size: dp(20)
                on_release: root.close_button_click()
""")


class EnteringReleaseDialog:
    """放流量の入力ダイアログボックスを定義したクラス。"""

    __slots__ = ["_view", "_layout", "_on_close_callback", "after_release"]

    def __init__(
        self,
        current_amount: int,
        atrer_release: int,
        sound_click: Sound,
        callback: Callable,
    ) -> None:
        """放流量の入力ダイアログボックスのインスタンスを作成して返す。

        Args:
            current_amount (int): 現在の貯水量。
            after_release (int): 放流後の貯水量。
            sound_click (Sound): クリック音のインスタンス。
            callback (Callable | None, optional): 「決定」ボタンがクリックされたときの処理。
        """

        size_hint: tuple[float, float] = (0.6, 0.4)
        pos_hint: dict[str, float] = {"x": 0.05, "top": 0.98}
        self._on_close_callback = callback
        self._view: ModalView = ModalView(
            size_hint=size_hint, pos_hint=pos_hint, auto_dismiss=False
        )
        self._layout: EnteringReleaseDialogLayout = EnteringReleaseDialogLayout(
            close_button_click=self._close,
            current_amount=current_amount,
            after_release=atrer_release,
            sound_click=sound_click,
        )
        self._view.add_widget(self._layout)

    def show(self) -> None:
        """MessageBox を表示する。"""

        self._view.open()

    def _close(self) -> None:
        """「閉じる」ボタンがクリックされたとき。"""

        self.after_release = self._layout.after_release
        """放流後の貯水量"""

        self._view.dismiss()

        self._on_close_callback()


class EnteringReleaseDialogLayout(AnchorLayout):
    """メッセージボックスのレイアウト。"""

    close_button_click: Callable = ObjectProperty(None)
    """「閉じる」ボタンがクリックされたときの処理を行う関数"""

    current_amount: int = NumericProperty(0)
    """現在の貯水量"""

    after_release: int = NumericProperty(0)
    """放流後の貯水量"""

    sound_click: Sound = ObjectProperty(None)
    """クリック音のインスタンス"""

    def __init__(self, **kwargs) -> None:
        super().__init__(**kwargs)

        self._increment_button: Button = self.ids.increment_button
        """「+」ボタンのインスタンス"""
        self._decrement_button: Button = self.ids.decrement_button
        """「-」ボタンのインスタンス"""
        self._post_release_volume_label: Label = self.ids.post_release_volume_label
        """放流後の貯水量を表示する Label ウィジェットのインスタンス"""

        self._keyboard = Window.request_keyboard(self._keyboard_closed, self)
        self._keyboard.bind(on_key_down=self._on_keyboard_down)

        if self.current_amount == 1:
            self.after_release = 0
            self._increment_button.disabled = True
            self._decrement_button.disabled = True
        else:
            self._increment_button.disabled = False
            self._decrement_button.disabled = False

        amount: int = int(self.after_release * 100)
        self._post_release_volume_label.text = f"{amount:,} 万㎥"

    def _keyboard_closed(self):
        """キーボードが閉じられたときに呼び出されるコールバック。"""

        self._keyboard.unbind(on_key_down=self._on_keyboard_down)

    def _on_keyboard_down(
        self,
        keyboard: Keyboard,
        keycode: tuple[int, str],
        text: str,
        modifiers: list[str],
    ) -> None:
        """押下されたアクセスキーにより処理を行う。

        Args:
            keyboard (Keyboard): バインドされた Keyboard のインスタンス。
            keycode (tupl[int, str]): 押されたキーのキーコードと文字の tuple。
            text (str): 押されたキーのテキスト。
            modifiers (list[str]): 同時に押された補助キー名の list。
        """

        if keycode[1].lower() in ("spacebar", "enter", "escape"):
            self._keyboard.unbind(on_key_down=self._on_keyboard_down)
            self.close_button_click()

    def on_increment_button_click(self) -> None:
        """「+」ボタンがクリックされたとき。"""

        if self.after_release < (self.current_amount - 1):
            self.after_release += 1
            amount: int = int(self.after_release * 100)
            self._post_release_volume_label.text = f"{amount:,} 万㎥"
            self.sound_click.play()

    def on_decrement_button_click(self) -> None:
        """「-」ボタンがクリックされたとき。"""

        if self.after_release > 1:
            self.after_release -= 1
            amount: int = int(self.after_release * 100)
            self._post_release_volume_label.text = f"{amount:,} 万㎥"
            self.sound_click.play()
