"""
カード一覧画面を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import math
from typing import Callable, override

from kivy.input.motionevent import MotionEvent
from kivy.lang import Builder
from kivy.metrics import dp
from kivy.uix.floatlayout import FloatLayout
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.uix.screenmanager import Screen

from models.card_type import CardType
from models.game_scene import GameScene
from models.my_config import MyConfig
from models.shared_vars import SharedVars

Builder.load_string("""
<CardsScreen>:
    BoxLayout:
        orientation: 'vertical'
        canvas.before:
            Color:
                rgba: 144/256, 238/256, 144/256, 1.0
            Rectangle:
                pos: self.pos
                size: self.size
        AnchorLayout:
            anchor_x: 'center'
            anchor_y: 'top'
            size_hint_y: 0.1
            Label:
                text: 'カード一覧'
                font_size: dp(30)
                color: 0.0, 0.3, 0.0, 1
        FloatLayout:
            id: card_space
            size_hint_y: 0.9
""")


class CardsScreen(Screen):
    """カード一覧画面を管理するクラス。"""

    def __init__(
        self,
        config: MyConfig,
        shared_vars: SharedVars,
        to_next_screen: Callable[[GameScene], None],
        **kwargs,
    ) -> None:
        """カード一覧画面を管理するクラスのインスタンスを作成して返す。

        Args:
            config (MyConfig): アプリの設定値を管理するクラスのインスタンス。
            shared_vars (SharedVars): アプリ内で共有する変数を管理するクラスのインスタンス。
            to_next_screen (Callable[[GameScene], None]): 次の画面に遷移するための関数。
        """

        super().__init__(**kwargs)

        self._config = config
        """アプリの設定値を管理するクラスのインスタンス"""

        self._shared_vars = shared_vars
        """アプリ内で共有する変数を管理するクラスのインスタンス"""

        self._to_next_screen: Callable[[GameScene], None] = to_next_screen
        """次の画面に遷移するための関数"""

        self._card_space: FloatLayout = self.ids.card_space
        """カード表示領域の FloatLayout"""

    @override
    def on_pre_enter(self, *_) -> None:
        """画面が表示される前の処理。"""

        super().on_pre_enter()

        self._card_space.clear_widgets()
        self._display_cards()

    def _display_cards(self) -> None:
        """カード一覧を表示する。"""

        rows: float = math.ceil(len(CardType) / 2)
        size_hint: float = 1 / rows - 0.01

        for i, type in enumerate(CardType):
            col: int = i % 2
            row: int = rows - 1 - (i // 2)
            x, y = (col * 0.5), (row * 0.105 + 0.04)
            pos_hint: dict[str, float] = {"x": x, "y": y}
            image: Image = self._get_image(type, size_hint, pos_hint)
            self._card_space.add_widget(image)

            x, y = (col * 0.5 - 0.38), (row * 0.105 - 0.45 + 0.04)
            pos_hint = {"x": x, "y": y}
            number: Label = self._get_number(type, pos_hint)
            self._card_space.add_widget(number)

            x, y = (col * 0.5 - 0.19), (row * 0.105 - 0.45 + 0.04)
            pos_hint = {"x": x, "y": y}
            name_and_description: Label = self._get_name_and_description(type, pos_hint)
            self._card_space.add_widget(name_and_description)

    def _get_image(
        self,
        type: CardType,
        size_hint: float,
        pos_hint: dict[str, float],
    ) -> Image:
        """カードの画像の Image を返す。

        Args:
            type (CardType): カードの種類。
            size_hint (float): 画像の大きさ。
            pos_hint (dict[str, float]): 画像の表示位置。

        Returns:
            Image: カードの画像の Image。
        """

        if self._config:
            is_card_added: bool = self._config.add_option_cards
            if (not is_card_added) and (type == CardType.TYPHOON):
                path: str = "card_blank.png"
            else:
                path: str = type.image_filename
        else:
            path: str = "card_blank.png"

        return Image(source=path, size_hint=(size_hint, size_hint), pos_hint=pos_hint)

    def _get_number(self, type: CardType, pos_hint: dict[str, float]) -> Label:
        """カードの枚数が入力された Label を返す。

        Args:
            type (CardType): カードの種類。
            pos_hint (dict[str, float]): Label の表示位置。

        Returns:
            Label: カードの枚数が入力された Label。
        """

        is_card_added: bool = self._config.add_option_cards
        if self._config and (not is_card_added) and (type == CardType.TYPHOON):
            text: str = ""
        else:
            text: str = f"{type.count} 枚"

        color = (0.0, 0.3, 0.0, 1)
        label: Label = Label(
            text=text,
            color=color,
            pos_hint=pos_hint,
            size=(dp(50), dp(60)),
            text_size=(dp(50), dp(60)),
            halign="left",
            valign="middle",
            font_size=dp(20),
        )
        return label

    def _get_name_and_description(
        self, type: CardType, pos_hint: dict[str, float]
    ) -> Label:
        """カードの名前と説明が入力された Label を返す。

        Args:
            type (CardType): カードの種類。
            pos_hint (dict[str, float]): Label の表示位置。

        Returns:
            Label: カードの名前と説明が入力された Label。
        """

        is_card_added: bool = self._config.add_option_cards
        if self._config and (not is_card_added) and (type == CardType.TYPHOON):
            text: str = ""
        else:
            description = type.description.replace("または", "\n　または")
            text: str = f"【{type.card_name}】\n　{description}"

        color = (0.0, 0.3, 0.0, 1)
        label: Label = Label(
            text=text,
            color=color,
            font_size=dp(12),
            pos_hint=pos_hint,
            size=(dp(270), dp(60)),
            text_size=(dp(270), dp(60)),
            halign="left",
            valign="middle",
            line_height=1.2,
        )
        return label

    @override
    def on_touch_down(self, touch: MotionEvent) -> None:
        """画面がタッチされたときの処理。

        Args:
            touch (MotionEvent): タッチイベント。
        """

        super().on_touch_down(touch)

        self._to_next_screen(self._shared_vars.prev_scene)
