"""
BGM / 効果音の再生シーンの定義。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from enum import IntEnum, auto


class SoundScene(IntEnum):
    """BGM / 効果音の種類の定義。"""

    OPENING = auto()
    """オープニング BGM"""

    NORMAL_MODE = auto()
    """「ふつう」モードでプレイ中の BGM"""

    HARD_MODE = auto()
    """「ちょいむず」モードでプレイ中の BGM"""

    WINNING_EXIT = auto()
    """優勝で終わったときの BGM"""

    LOSE_EXIT = auto()
    """優勝できずに終わったときの BGM"""

    DRAW_EXIT = auto()
    """引き分けで終わったときの BGM"""

    DROP_OUT_EXIT = auto()
    """失格で終わったときの BGM"""

    SCREEN_TRANSITION = auto()
    """画面遷移時の SE"""

    CLICK = auto()
    """クリック時の SE"""

    CARD_FLIP = auto()
    """カードをめくるときの SE"""

    DROP_OUT = auto()
    """失格時の SE"""

    RECOVERY = auto()
    """復帰時の SE"""

    @property
    def filename(self) -> str:
        """音声ファイルの名前を返す。"""

        match self:
            case SoundScene.OPENING:
                return "tomin_shuryo.mp3"
            case SoundScene.NORMAL_MODE:
                return "maou_bgm_orchestra24.mp3"
            case SoundScene.HARD_MODE:
                return "maou_bgm_orchestra25.mp3"
            case SoundScene.WINNING_EXIT:
                return "tassei.mp3"
            case SoundScene.LOSE_EXIT:
                return "maou_se_jingle02.mp3"
            case SoundScene.DRAW_EXIT:
                return "maou_se_jingle01.mp3"
            case SoundScene.DROP_OUT_EXIT:
                return "maou_se_jingle13.mp3"
            case SoundScene.SCREEN_TRANSITION:
                return "maou_se_onepoint15.mp3"
            case SoundScene.CLICK:
                return "maou_se_system40.mp3"
            case SoundScene.CARD_FLIP:
                return "maou_se_sound_ignition01.mp3"
            case SoundScene.DROP_OUT:
                return "maou_se_system08.mp3"
            case SoundScene.RECOVERY:
                return "maou_se_onepoint09.mp3"

    @property
    def volume_scale(self) -> float:
        """音量の調整率を返す。"""

        match self:
            case SoundScene.OPENING:
                return 1.0
            case SoundScene.NORMAL_MODE:
                return 1.0
            case SoundScene.HARD_MODE:
                return 1.5
            case SoundScene.WINNING_EXIT:
                return 1.0
            case SoundScene.LOSE_EXIT:
                return 1.0
            case SoundScene.DRAW_EXIT:
                return 1.0
            case SoundScene.DROP_OUT_EXIT:
                return 1.0
            case SoundScene.SCREEN_TRANSITION:
                return 1.0
            case SoundScene.CLICK:
                return 1.0
            case SoundScene.CARD_FLIP:
                return 1.0
            case SoundScene.DROP_OUT:
                return 6.0
            case SoundScene.RECOVERY:
                return 6.0

    @property
    def is_bgm(self) -> bool:
        """ループ再生する BGM であれば True を返す。"""

        return self in {
            SoundScene.OPENING,
            SoundScene.NORMAL_MODE,
            SoundScene.HARD_MODE,
        }

    @property
    def title(self) -> str:
        """音声ファイルのタイトルを返す。"""

        match self:
            case SoundScene.OPENING:
                return "冬眠終了！"
            case SoundScene.NORMAL_MODE:
                return "オーケストラ24"
            case SoundScene.HARD_MODE:
                return "オーケストラ25"
            case SoundScene.WINNING_EXIT:
                return "達成"
            case SoundScene.LOSE_EXIT:
                return "ジングル02"
            case SoundScene.DRAW_EXIT:
                return "ジングル01"
            case SoundScene.DROP_OUT_EXIT:
                return "ジングル13"
            case SoundScene.SCREEN_TRANSITION:
                return "ワンポイント15"
            case SoundScene.CLICK:
                return "システム40"
            case SoundScene.CARD_FLIP:
                return "点火01"
            case SoundScene.DROP_OUT:
                return "システム08"
            case SoundScene.RECOVERY:
                return "ワンポイント09"

    @property
    def site_name(self) -> str:
        """音声ファイルの配布元のサイト名を返す。"""

        match self:
            case SoundScene.OPENING:
                return "こんとどぅふぇ"
            case SoundScene.NORMAL_MODE:
                return "魔王魂"
            case SoundScene.HARD_MODE:
                return "魔王魂"
            case SoundScene.WINNING_EXIT:
                return "DOVA-SYNDROME"
            case SoundScene.LOSE_EXIT:
                return "魔王魂"
            case SoundScene.DRAW_EXIT:
                return "魔王魂"
            case SoundScene.DROP_OUT_EXIT:
                return "魔王魂"
            case SoundScene.SCREEN_TRANSITION:
                return "魔王魂"
            case SoundScene.CLICK:
                return "魔王魂"
            case SoundScene.CARD_FLIP:
                return "魔王魂"
            case SoundScene.DROP_OUT:
                return "魔王魂"
            case SoundScene.RECOVERY:
                return "魔王魂"

    @property
    def site_url(self) -> str:
        """音声ファイルの配布元の URL を返す。"""

        match self:
            case SoundScene.OPENING:
                return "https://conte-de-fees.com/"
            case SoundScene.NORMAL_MODE:
                return "https://maou.audio/"
            case SoundScene.HARD_MODE:
                return "https://maou.audio/"
            case SoundScene.WINNING_EXIT:
                return "https://dova-s.jp/"
            case SoundScene.LOSE_EXIT:
                return "https://maou.audio/"
            case SoundScene.DRAW_EXIT:
                return "https://maou.audio/"
            case SoundScene.DROP_OUT_EXIT:
                return "https://maou.audio/"
            case SoundScene.SCREEN_TRANSITION:
                return "https://maou.audio/"
            case SoundScene.CLICK:
                return "https://maou.audio/"
            case SoundScene.CARD_FLIP:
                return "https://maou.audio/"
            case SoundScene.DROP_OUT:
                return "https://maou.audio/"
            case SoundScene.RECOVERY:
                return "https://maou.audio/"
