"""
プレイヤーの情報を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from dataclasses import dataclass, field
from typing import Optional

import constants
from models.card import Card
from views.player_status import PlayerStatus


@dataclass(slots=True)
class Player:
    """プレイヤーの情報を管理するクラス。"""

    index: int
    """ウィンドウ上の表示位置のインデックス番号"""

    name: str
    """名前"""

    is_human: bool
    """人間が操作するプレイヤーであれば True"""

    base_point: dict[str, float]
    """レイアウトの基準座標"""

    status: PlayerStatus
    """プレイヤーの状態を表示するウィジェット"""

    water_storage: int = constants.WATER_STORAGE_AT_START
    """貯水量"""

    is_drop_out: bool = False
    """失格した場合は True"""

    is_win: bool = False
    """勝利した場合は True"""

    is_draw: bool = False
    """引き分けの場合は True"""

    cards_in_hand: list[Card] = field(default_factory=list)
    """手札の list"""

    next_player: Optional["Player"] = None
    """次の順番のプレイヤー"""

    prev_player: Optional["Player"] = None
    """前の順番のプレイヤー"""

    def update_status(self, marker_text: str, rank_text: str) -> None:
        """プレイヤーの情報を更新する。

        Args:
            marker_text (str): マーカーの文字列
            rank_text (str): 順位の文字列
        """

        self.status.maker_text = marker_text
        self.status.name_text = self.name
        self.status.amount_text = f"貯水量 {self.water_storage * 100} 万㎥"
        self.status.rank_text = rank_text

        if marker_text == "順番":
            self.status.maker_text_color = [1.0, 1.0, 1.0, 1.0]
            self.status.marker_bg_color = [1.0, 0.0, 0.0, 1.0]
        elif marker_text == "失格":
            self.status.maker_text_color = [1.0, 1.0, 1.0, 1.0]
            self.status.marker_bg_color = [0.0, 0.0, 0.0, 1.0]
            self.status.name_text_color = [0.5, 0.5, 0.5, 1.0]
            self.status.amount_text_color = [0.5, 0.5, 0.5, 1.0]
        elif marker_text == "優勝":
            self.status.maker_text_color = [7.0, 0.0, 0.0, 1.0]
            self.status.marker_bg_color = [1.0, 1.0, 0.0, 1.0]
        else:
            self.status.maker_text_color = [0.0, 0.0, 0.0, 1.0]
            self.status.marker_bg_color = [0.0, 0.0, 0.0, 0.0]

        if rank_text == "越水":
            self.status.amount_bg_color = [1.0, 0.7, 0.7, 1.0]
            self.status.rank_text_color = [0.8, 0.0, 0.0, 1.0]
            self.status.rank_bg_color = [1.0, 0.7, 0.7, 1.0]
        elif rank_text == "渇水":
            self.status.amount_bg_color = [1.0, 1.0, 0.5, 1.0]
            self.status.rank_text_color = [0.8, 0.0, 0.0, 1.0]
            self.status.rank_bg_color = [1.0, 1.0, 0.5, 1.0]
        elif rank_text == "最多":
            self.status.amount_bg_color = [0.7, 0.7, 1.0, 1.0]
            self.status.rank_text_color = [0.0, 0.0, 0.7, 1.0]
            self.status.rank_bg_color = [0.7, 0.7, 1.0, 1.0]
        elif rank_text == "２位":
            self.status.amount_bg_color = [0.9, 0.9, 1.0, 1.0]
            self.status.rank_text_color = [0.0, 0.0, 0.7, 1.0]
            self.status.rank_bg_color = [0.9, 0.9, 1.0, 1.0]
        else:
            self.status.amount_bg_color = [0.0, 0.0, 0.0, 0.0]
            self.status.rank_text_color = [0.0, 0.0, 0.0, 0.0]
            self.status.rank_bg_color = [0.0, 0.0, 0.0, 0.0]
