"""
カードの種類の定義。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from enum import IntEnum, auto

from models.affected_player import AffectedPlayer
from models.affected_target import AffectedTarget


class CardType(IntEnum):
    """カードの種類の定義。"""

    RAINY_WEATHER = auto()
    """雨 (全)"""

    SHOWER_RAIN_N = auto()
    """にわか雨 (次)"""

    SHOWER_RAIN_L = auto()
    """にわか雨 (多)"""

    HEAVY_RAIN = auto()
    """大雨 (全)"""

    THUNDERSTORM_N = auto()
    """雷雨 (次)"""

    THUNDERSTORM_L = auto()
    """雷雨 (多)"""

    TORRENTIAL_RAIN_N = auto()
    """集中豪雨 (次)"""

    TORRENTIAL_RAIN_L = auto()
    """集中豪雨 (多)"""

    TORRENTIAL_RAIN_S = auto()
    """集中豪雨 (自)"""

    TYPHOON = auto()
    """台風（少）"""

    DROUGHT_A = auto()
    """日照り (全)"""

    DROUGHT_L = auto()
    """日照り (多)"""

    RELEASE = auto()
    """放流"""

    REVERSE = auto()
    """リバース"""

    DRAW_OWN_1_CARD = auto()
    """自分の手札＋１"""

    DRAW_OWN_2_CARDS = auto()
    """自分の手札＋２"""

    DRAW_1_CARD_OF_ALL_PLAYERS = auto()
    """全員の手札＋１"""

    DRAW_2_CARDS_OF_ALL_PLAYERS = auto()
    """全員の手札＋２"""

    @property
    def card_name(self) -> str:
        """カードの名前を返す。"""

        dictionary: dict[CardType, str] = {
            CardType.RAINY_WEATHER: "雨 (全)",
            CardType.SHOWER_RAIN_N: "にわか雨 (次)",
            CardType.SHOWER_RAIN_L: "にわか雨 (多)",
            CardType.HEAVY_RAIN: "大雨 (全)",
            CardType.THUNDERSTORM_N: "雷雨 (次)",
            CardType.THUNDERSTORM_L: "雷雨 (多)",
            CardType.TORRENTIAL_RAIN_N: "集中豪雨 (次)",
            CardType.TORRENTIAL_RAIN_L: "集中豪雨 (多)",
            CardType.TORRENTIAL_RAIN_S: "集中豪雨 (自)",
            CardType.TYPHOON: "台風（少）",
            CardType.DROUGHT_A: "日照り (全)",
            CardType.DROUGHT_L: "日照り (多)",
            CardType.RELEASE: "放流",
            CardType.REVERSE: "リバース",
            CardType.DRAW_OWN_1_CARD: "自分の手札＋１",
            CardType.DRAW_OWN_2_CARDS: "自分の手札＋２",
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: "全員の手札＋１",
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: "全員の手札＋２",
        }

        return dictionary[self]

    @property
    def count(self) -> int:
        """カードの枚数を返す。"""

        dictionary: dict[CardType, int] = {
            CardType.RAINY_WEATHER: 4,
            CardType.SHOWER_RAIN_N: 5,
            CardType.SHOWER_RAIN_L: 3,
            CardType.HEAVY_RAIN: 4,
            CardType.THUNDERSTORM_N: 5,
            CardType.THUNDERSTORM_L: 3,
            CardType.TORRENTIAL_RAIN_N: 3,
            CardType.TORRENTIAL_RAIN_L: 3,
            CardType.TORRENTIAL_RAIN_S: 2,
            CardType.TYPHOON: 1,
            CardType.DROUGHT_A: 4,
            CardType.DROUGHT_L: 4,
            CardType.RELEASE: 8,
            CardType.REVERSE: 4,
            CardType.DRAW_OWN_1_CARD: 3,
            CardType.DRAW_OWN_2_CARDS: 1,
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: 3,
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: 1,
        }

        return dictionary[self]

    @property
    def affected_player(self) -> AffectedPlayer:
        """影響を受けるプレイヤーを返す。"""

        dictionary: dict[CardType, AffectedPlayer] = {
            CardType.RAINY_WEATHER: AffectedPlayer.ALL_PLAYERS,
            CardType.SHOWER_RAIN_N: AffectedPlayer.NEXT_PLAYER,
            CardType.SHOWER_RAIN_L: AffectedPlayer.PLAYER_WITH_MOST_WATER,
            CardType.HEAVY_RAIN: AffectedPlayer.ALL_PLAYERS,
            CardType.THUNDERSTORM_N: AffectedPlayer.NEXT_PLAYER,
            CardType.THUNDERSTORM_L: AffectedPlayer.PLAYER_WITH_MOST_WATER,
            CardType.TORRENTIAL_RAIN_N: AffectedPlayer.NEXT_PLAYER,
            CardType.TORRENTIAL_RAIN_L: AffectedPlayer.PLAYER_WITH_MOST_WATER,
            CardType.TORRENTIAL_RAIN_S: AffectedPlayer.SELF,
            CardType.TYPHOON: AffectedPlayer.PLAYER_WITH_LOWEST_WATER,
            CardType.DROUGHT_A: AffectedPlayer.ALL_PLAYERS,
            CardType.DROUGHT_L: AffectedPlayer.PLAYER_WITH_MOST_WATER,
            CardType.RELEASE: AffectedPlayer.SELF,
            CardType.REVERSE: AffectedPlayer.ALL_PLAYERS,
            CardType.DRAW_OWN_1_CARD: AffectedPlayer.SELF,
            CardType.DRAW_OWN_2_CARDS: AffectedPlayer.SELF,
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: AffectedPlayer.ALL_PLAYERS,
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: AffectedPlayer.ALL_PLAYERS,
        }

        return dictionary[self]

    @property
    def affected_target(self) -> AffectedTarget:
        """影響を受ける項目を返す。"""

        dictionary: dict[CardType, AffectedTarget] = {
            CardType.RAINY_WEATHER: AffectedTarget.WATER_STORAGE,
            CardType.SHOWER_RAIN_N: AffectedTarget.WATER_STORAGE,
            CardType.SHOWER_RAIN_L: AffectedTarget.WATER_STORAGE,
            CardType.HEAVY_RAIN: AffectedTarget.WATER_STORAGE,
            CardType.THUNDERSTORM_N: AffectedTarget.WATER_STORAGE,
            CardType.THUNDERSTORM_L: AffectedTarget.WATER_STORAGE,
            CardType.TORRENTIAL_RAIN_N: AffectedTarget.WATER_STORAGE,
            CardType.TORRENTIAL_RAIN_L: AffectedTarget.WATER_STORAGE,
            CardType.TORRENTIAL_RAIN_S: AffectedTarget.WATER_STORAGE,
            CardType.TYPHOON: AffectedTarget.WATER_STORAGE,
            CardType.DROUGHT_A: AffectedTarget.WATER_STORAGE,
            CardType.DROUGHT_L: AffectedTarget.WATER_STORAGE,
            CardType.RELEASE: AffectedTarget.WATER_STORAGE,
            CardType.REVERSE: AffectedTarget.NONE,
            CardType.DRAW_OWN_1_CARD: AffectedTarget.CARDS_IN_HAND,
            CardType.DRAW_OWN_2_CARDS: AffectedTarget.CARDS_IN_HAND,
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: AffectedTarget.CARDS_IN_HAND,
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: AffectedTarget.CARDS_IN_HAND,
        }

        return dictionary[self]

    @property
    def water_increase_amount(self) -> int:
        """貯水量の増加量を返す。"""

        dictionary: dict[CardType, int] = {
            CardType.RAINY_WEATHER: +1,
            CardType.SHOWER_RAIN_N: +1,
            CardType.SHOWER_RAIN_L: +1,
            CardType.HEAVY_RAIN: +3,
            CardType.THUNDERSTORM_N: +3,
            CardType.THUNDERSTORM_L: +3,
            CardType.TORRENTIAL_RAIN_N: +5,
            CardType.TORRENTIAL_RAIN_L: +5,
            CardType.TORRENTIAL_RAIN_S: +5,
            CardType.TYPHOON: +7,
            CardType.DROUGHT_A: -2,
            CardType.DROUGHT_L: -2,
            CardType.RELEASE: -1,
            CardType.REVERSE: 0,
            CardType.DRAW_OWN_1_CARD: 0,
            CardType.DRAW_OWN_2_CARDS: 0,
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: 0,
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: 0,
        }

        return dictionary[self]

    @property
    def draw_count(self) -> int:
        """山札から引くカードの枚数を返す。"""

        dictionary: dict[CardType, int] = {
            CardType.RAINY_WEATHER: 0,
            CardType.SHOWER_RAIN_N: 0,
            CardType.SHOWER_RAIN_L: 0,
            CardType.HEAVY_RAIN: 0,
            CardType.THUNDERSTORM_N: 0,
            CardType.THUNDERSTORM_L: 0,
            CardType.TORRENTIAL_RAIN_N: 0,
            CardType.TORRENTIAL_RAIN_L: 0,
            CardType.TORRENTIAL_RAIN_S: 0,
            CardType.TYPHOON: 0,
            CardType.DROUGHT_A: 0,
            CardType.DROUGHT_L: 0,
            CardType.RELEASE: 0,
            CardType.REVERSE: 0,
            CardType.DRAW_OWN_1_CARD: +1,
            CardType.DRAW_OWN_2_CARDS: +2,
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: +1,
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: +2,
        }

        return dictionary[self]

    @property
    def image_filename(self) -> str:
        """カードの画像ファイルの名前を返す。"""

        dictionary: dict[CardType, str] = {
            CardType.RAINY_WEATHER: "card_rainy_weather.png",
            CardType.SHOWER_RAIN_N: "card_shower_rain_n.png",
            CardType.SHOWER_RAIN_L: "card_shower_rain_l.png",
            CardType.HEAVY_RAIN: "card_heavy_rain.png",
            CardType.THUNDERSTORM_N: "card_thunderstorm_n.png",
            CardType.THUNDERSTORM_L: "card_thunderstorm_l.png",
            CardType.TORRENTIAL_RAIN_N: "card_torrential_rain_n.png",
            CardType.TORRENTIAL_RAIN_L: "card_torrential_rain_l.png",
            CardType.TORRENTIAL_RAIN_S: "card_torrential_rain_s.png",
            CardType.TYPHOON: "card_typhoon.png",
            CardType.DROUGHT_A: "card_drought_a.png",
            CardType.DROUGHT_L: "card_drought_l.png",
            CardType.RELEASE: "card_discharge.png",
            CardType.REVERSE: "card_reverse.png",
            CardType.DRAW_OWN_1_CARD: "card_draw_own_1card.png",
            CardType.DRAW_OWN_2_CARDS: "card_draw_own_2cards.png",
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: "all_players_each_draw.png",
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: "all_players_2cards_draw.png",
        }

        return dictionary[self]

    @property
    def description(self) -> str:
        """カードの説明文を返す。"""

        dictionary: dict[CardType, str] = {
            CardType.RAINY_WEATHER: "全員の貯水量を 100万㎥ 増やす。",
            CardType.SHOWER_RAIN_N: "次の順番の人の貯水量を 100万㎥ 増やす。",
            CardType.SHOWER_RAIN_L: "貯水量が最多の人の貯水量を 100万㎥ 増やす。",
            CardType.HEAVY_RAIN: "全員の貯水量を 300万㎥ 増やす。",
            CardType.THUNDERSTORM_N: "次の順番の人の貯水量を 300万㎥ 増やす。",
            CardType.THUNDERSTORM_L: "貯水量が最多の人の貯水量を 300万㎥ 増やす。",
            CardType.TORRENTIAL_RAIN_N: "次の順番の人の貯水量を 500万㎥ 増やす。",
            CardType.TORRENTIAL_RAIN_L: "貯水量が最多の人の貯水量を 500万㎥ 増やす。",
            CardType.TORRENTIAL_RAIN_S: "自分の貯水量を 500万㎥ 増やす。または、渇水 から復帰する。",
            CardType.TYPHOON: "貯水量が最少の人の貯水量を 700万㎥ 増やす。",
            CardType.DROUGHT_A: "全員の貯水量を 200万㎥ 減らす。",
            CardType.DROUGHT_L: "貯水量が最多の人の貯水量を 200万㎥ 減らす。",
            CardType.RELEASE: "自分の貯水量を 好きなだけ 減らす。または、越水 から復帰する。",
            CardType.REVERSE: "カードを出す順番を逆回りにする。",
            CardType.DRAW_OWN_1_CARD: "自分が山札からカードを 1 枚引く。",
            CardType.DRAW_OWN_2_CARDS: "自分が山札からカードを 2 枚引く。",
            CardType.DRAW_1_CARD_OF_ALL_PLAYERS: "全員が山札からカードを 1 枚引く。",
            CardType.DRAW_2_CARDS_OF_ALL_PLAYERS: "全員が山札からカードを 2 枚引く。",
        }

        return dictionary[self]
