"""
「放流」に関する戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import random

import models.situation_in_hand as situation_in_hand
from controllers.tactics.common_tactics import CommonTactics
from models.card import Card
from models.card_type import CardType
from models.player import Player


def get_water_amount_after_release(
    common_tactics: CommonTactics, player: Player, players: list[Player]
) -> int:
    """コンピューターが操作するプレイヤーの放流後の貯水量、または、人間が操作するプレイヤーが
    放流後の貯水量を入力する際の初期値を算出して返す。

    Args:
        common_tactics (CommonTactics): 各モード共通のコンピューターの戦術。
        player (Player): 対象プレイヤー。
        players (list[Player]): 失格になっていないすべてのプレイヤー。

    Returns:
        int: 放流後の貯水量。
    """

    amount: int = player.water_storage

    if amount == 1:
        remaining: int = 0

    else:
        second: int = common_tactics.get_2nd_largest_water_amount(players)

        if common_tactics.is_late_stage(players):
            remaining = _get_remaining_at_late_stage(
                common_tactics, player, players, second
            )
        else:
            remaining = _get_remaining_at_not_late_stage(
                common_tactics, player, players, second
            )

    return remaining


def _get_remaining_at_not_late_stage(
    ct: CommonTactics, player: Player, players: list[Player], second: int
) -> int:
    """ゲームの序盤から中盤における、放流後の貯水量を算出して返す。

    Args:
        ct (CommonTactics): 各モード共通のコンピューターの戦術。
        player (Player): 対象プレイヤー。
        players (list[Player]): 失格になっていないすべてのプレイヤー。
        second (int): 対象プレイヤーの貯水量が最多のときの 2 位の貯水量。

    Returns:
        int: 放流後の貯水量。
    """

    amount: int = player.water_storage
    in_hand: list[Card] = player.cards_in_hand
    are_amounts_equal: bool = ct.are_all_water_amounts_equal(players)
    has_card: bool = situation_in_hand.has_specified_card(in_hand, [CardType.DROUGHT_L])

    if (len(players) == 2) and are_amounts_equal and has_card:
        remaining: int = amount - 1
    elif ct.is_largest_water_amount(player, players):
        target: int = second - 1 - int(random.normalvariate(0.0, 0.8))
        remaining: int = min([max([target, 1]), amount - 1])
    else:
        remaining: int = amount - 1

    return remaining


def _get_remaining_at_late_stage(
    ct: CommonTactics, player: Player, players: list[Player], second: int
) -> int:
    """ゲーム終盤における、放流後の貯水量を算出して返す。

    Args:
        ct (CommonTactics): 各モード共通のコンピューターの戦術。
        player (Player): 対象プレイヤー。
        players (list[Player]): 失格になっていないすべてのプレイヤー。
        second (int): 対象プレイヤーの貯水量が最多のときの 2 位の貯水量。

    Returns:
        int: 放流後の貯水量。
    """

    amount: int = player.water_storage
    largest: int = ct.get_largest_water_amount(players)
    in_hand: list[Card] = player.cards_in_hand
    card_types: list[CardType] = [CardType.TORRENTIAL_RAIN_S]

    if situation_in_hand.has_specified_card(in_hand, card_types):
        if ct.is_largest_water_amount(player, players):
            target: int = _get_target_amount(ct, second)
        else:
            target: int = _get_target_amount(ct, largest)

        remaining: int = min([max([(target - 5), 1]), (amount - 1)])
    else:
        if ct.are_all_water_amounts_equal(players):
            remaining: int = amount - 1
        elif ct.is_largest_water_amount(player, players):
            target: int = _get_target_amount(ct, second)
            remaining: int = min([max([target, 1]), (amount - 1)])
        else:
            remaining: int = amount - 1

    return remaining


def _get_target_amount(self, amount: int) -> int:
    """ゲーム終了時の最終目標となる目安の貯水量を算出して返す。

    Args:
        amount (int): 対象プレイヤーがトップの場合は２位の貯水量、トップでない場合はトップの貯水量。

    Returns:
        int: 最終目標の目安となる貯水量。
    """

    if (amount + 6) <= 10:
        return amount + 6
    elif (amount + 6) <= 15:
        offset: int = max([min([(3 + int(random.normalvariate(0, 1.2))), 6]), 1])
        return amount + offset
    elif (amount + 4) <= 15:
        offset: int = max([min([(2 + int(random.normalvariate(0, 1.0))), 4]), 1])
        return amount + offset
    else:
        return amount + 1
