"""
コンピューターが操作するプレイヤーの手札の出し方に関する戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

from models.card import Card
from models.card_type import CardType


def get_card_by_priority(cards: list[Card], card_types: list[CardType]) -> Card:
    """card_types で指定された優先順位に従って、手札の list から一致する種類のカードを１枚取得して返す
    （card_types で指定された種類のカードが、必ず cards に含まれている前提）。

    Args:
        cards (list[Card]): 手札の list。
        card_types (list[CardType]): カードの種類（優先順）。

    Returns:
        Card: 手札の中の一致するカード。
    """

    for type in card_types:
        if any(ext := [c for c in cards if (c.type is type)]):
            return ext[0]

    return cards[0]  # ダミーの戻り値


def get_non_specified_types_cards(
    cards: list[Card], types: list[CardType]
) -> list[Card]:
    """指定された種類以外のカードを手札から抽出して返す。

    Args:
        cards (list[Card]): 手札の list。
        types (list[CardType]): カードの種類。

    Returns:
        list[Card]: 手札から抽出されたカード。
    """

    return [c for c in cards if (c.type not in types)]


def get_cards_in_other_cases(
    in_hand: list[Card], water_storage: int, is_largest: bool
) -> list[Card]:
    """各モード共通の「その他のケース」において、できるだけ適切でない手札を除いた手札の list を返す。

    Args:
        in_hand (list[Card]): 手札の list。
        water_storage (int): 対象プレイヤーの貯水量。
        is_largest (bool): 対象プレイヤーの貯水量が最多であれば True。

    Returns:
        list[Card]: できるだけ適切でないカードを除いた手札の list。
    """

    if (water_storage == 2) and is_largest:
        types: list[CardType] = [
            CardType.DROUGHT_L,
            CardType.DROUGHT_A,
        ]
    elif water_storage == 2:
        types: list[CardType] = [CardType.DROUGHT_A]
    elif water_storage == 1:
        types: list[CardType] = [
            CardType.DROUGHT_A,
            CardType.DROUGHT_L,
            CardType.RELEASE,
        ]
    elif water_storage == 15:
        types: list[CardType] = [
            CardType.RAINY_WEATHER,
            CardType.SHOWER_RAIN_L,
            CardType.HEAVY_RAIN,
            CardType.THUNDERSTORM_L,
            CardType.TORRENTIAL_RAIN_L,
            CardType.TORRENTIAL_RAIN_S,
        ]
    elif (water_storage >= 13) and is_largest:
        types: list[CardType] = [
            CardType.HEAVY_RAIN,
            CardType.THUNDERSTORM_L,
            CardType.TORRENTIAL_RAIN_L,
            CardType.TORRENTIAL_RAIN_S,
        ]
    elif water_storage >= 13:
        types: list[CardType] = [CardType.HEAVY_RAIN, CardType.TORRENTIAL_RAIN_S]
    elif (water_storage >= 11) and is_largest:
        types: list[CardType] = [CardType.TORRENTIAL_RAIN_L, CardType.TORRENTIAL_RAIN_S]
    elif water_storage >= 11:
        types: list[CardType] = [CardType.TORRENTIAL_RAIN_S]
    elif not is_largest:
        types: list[CardType] = [CardType.DROUGHT_A, CardType.RELEASE]
    else:
        return []

    return get_non_specified_types_cards(in_hand, types)
