"""
コンピューターが操作するプレイヤーの戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging

import controllers.tactics.to_release as to_release
from controllers.tactics.common_tactics import CommonTactics
from controllers.tactics.in_2players_mode import In2PlayersMode
from controllers.tactics.in_hard_mode import InHardMode
from controllers.tactics.in_normal_mode import InNormalMode
from models.card import Card
from models.player import Player


def select_cards_to_exchange(
    player: Player, player_count: int, is_hard_mode: bool, logger: logging.Logger
) -> list[Card]:
    """コンピューターが操作するプレイヤーが、交換する手札を選択して、その手札の list を返す。

    Args:
        player (Player): 対象プレイヤー。
        player_count (int): プレイ人数。
        is_hard_mode (bool): 「ちょいムズ」モードの場合は True。
        logger (logging.Logger): logging.Logger のインスタンス。

    Returns:
        list[Card]: 交換する手札の list。
    """

    if player_count == 2:
        cards: list[Card] = In2PlayersMode(logger).select_cards_to_exchange(player)
    elif is_hard_mode:
        cards: list[Card] = InHardMode(logger).select_cards_to_exchange(player)
    else:
        cards: list[Card] = InNormalMode(logger).select_cards_to_exchange(player)

    return cards


def select_card_to_play(
    player: Player,
    all_players: list[Player],
    player_count: int,
    is_hard_mode: bool,
    logger: logging.Logger,
) -> Card:
    """コンピューターが操作するプレイヤーが、場に出すカードを選択して返す。

    Args:
        player (Player): 対象となるプレイヤー。
        all_players (list[Player]): すべてのプレイヤー。
        player_count (int): プレイ人数。
        is_hard_mode (bool): 「ちょいムズ」モードの場合は True。
        logger (logging.Logger): logging.Logger のインスタンス。

    Returns:
        Card: 場に出すカード。
    """

    players: list[Player] = [p for p in all_players if (not p.is_drop_out)]

    if player_count == 2:
        card: Card = In2PlayersMode(logger).select_card_to_play(player, players)
    elif is_hard_mode:
        card: Card = InHardMode(logger).select_card_to_play(player, players)
    else:
        card: Card = InNormalMode(logger).select_card_to_play(player, players)

    return card


def get_amount_of_water_after_release(
    player: Player, all_players: list[Player], logger: logging.Logger
) -> int:
    """コンピューターが操作するプレイヤーの放流後の貯水量、または、人間が操作するプレイヤーが
    放流後の貯水量を入力する際の初期値を算出して返す。

    Args:
        player (Player): 対象となるプレイヤー。
        all_players (list[Player]): すべてのプレイヤー。
        logger (logging.Logger): logging.Logger のインスタンス。

    Returns:
        int: 放流後の貯水量。
    """

    common_tactics: CommonTactics = CommonTactics(logger)
    players: list[Player] = [p for p in all_players if (not p.is_drop_out)]
    return to_release.get_water_amount_after_release(common_tactics, player, players)
