"""
「ちょいムズ」モード時のコンピューターの戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import math
import random

import controllers.tactics.to_exchange_cards as to_exchange_cards
import controllers.tactics.to_play_card as to_play_card
from controllers.tactics.common_tactics import CommonTactics
from models.card import Card
from models.card_type import CardType
from models.player import Player


class InHardMode:
    """「ちょいムズ」モード時のコンピューターの戦術。"""

    __slots__ = ["_logger", "_ct"]

    def __init__(self, logger: logging.Logger) -> None:
        """「ちょいムズ」モード時のコンピューターの戦術。

        Args:
            logger (logging.Logger): logging.Logger のインスタンス。
        """

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

        self._ct: CommonTactics = CommonTactics(self._logger)
        """各モード共通のコンピューターの戦術"""

    def select_cards_to_exchange(self, player: Player) -> list[Card]:
        """「ちょいムズ」モードのときに、交換する手札を選択して、その手札の list を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            list[Card]: 手札の list。
        """

        cards: list[Card] = player.cards_in_hand

        # 次または前の順番が人間プレイヤーの場合...
        if self._is_next_player_human(player) or self._is_prev_player_human(player):
            types: dict[CardType, int] = {
                CardType.TORRENTIAL_RAIN_N: 5,
                CardType.THUNDERSTORM_N: 5,
                CardType.SHOWER_RAIN_N: 5,
                CardType.RELEASE: 2,
                CardType.TORRENTIAL_RAIN_S: 1,
            }
            cards = to_exchange_cards.leave_target_cards(cards, types)
        else:
            types: dict[CardType, int] = {
                CardType.RELEASE: 2,
                CardType.TORRENTIAL_RAIN_S: 1,
                CardType.DRAW_OWN_2_CARDS: 1,
            }
            cards = to_exchange_cards.leave_target_cards(cards, types)

        return cards

    def _is_next_player_human(self, player: Player) -> bool:
        """指定されたプレイヤーの次の順番が、人間が操作するプレイヤーの場合は True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 次の順番のプレイヤーが人間が操作するプレイヤーであれば True。
        """

        return player.next_player.is_human  # type: ignore

    def _is_prev_player_human(self, player: Player) -> bool:
        """指定されたプレイヤーの前の順番が、人間が操作するプレイヤーの場合は True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 前の順番のプレイヤーが人間が操作するプレイヤーであれば True。
        """

        return player.prev_player.is_human  # type: ignore

    def select_card_to_play(self, player: Player, players: list[Player]) -> Card:
        """「ちょいムズ」モードのときに、場に出す手札を選択して返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            Card: 場に出すカード。
        """

        in_hand: list[Card] = player.cards_in_hand

        if (c := self._ct.get_card_in_common_cases(player, players)) is not None:
            card: Card = c
        elif self._does_case_01_hold(player, players):
            types: list[CardType] = [
                CardType.RELEASE,
                CardType.DROUGHT_A,
                CardType.DROUGHT_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_02_hold(player, players):
            types: list[CardType] = [CardType.DROUGHT_A, CardType.DROUGHT_L]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_03_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_04_hold(player, players):
            types: list[CardType] = [
                CardType.TORRENTIAL_RAIN_N,
                CardType.THUNDERSTORM_N,
                CardType.SHOWER_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_05_hold(player, players):
            types: list[CardType] = [
                CardType.TORRENTIAL_RAIN_L,
                CardType.THUNDERSTORM_L,
                CardType.HEAVY_RAIN,
                CardType.SHOWER_RAIN_L,
                CardType.RAINY_WEATHER,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_06_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_07_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_08_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_09_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_10_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_11_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_S]
            )
        elif self._does_case_12_hold(player, players):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_N,
                CardType.THUNDERSTORM_N,
                CardType.TORRENTIAL_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_13_hold(player, players):
            types: list[CardType] = [CardType.SHOWER_RAIN_N, CardType.THUNDERSTORM_N]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_14_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.SHOWER_RAIN_N]
            )
        elif self._does_case_15_hold(player, players):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_L,
                CardType.THUNDERSTORM_L,
                CardType.TORRENTIAL_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_16_hold(player, players):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_L,
                CardType.THUNDERSTORM_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_17_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.SHOWER_RAIN_L]
            )
        else:
            water_storage: int = player.water_storage
            is_largest: bool = self._ct.is_largest_water_amount(player, players)
            other_cards: list[Card] = to_play_card.get_cards_in_other_cases(
                in_hand, water_storage, is_largest
            )
            if any(other_cards):
                text: str = f"{player.name}の戦術："
                self._logger.info(f"{text}「その他のケース」に該当しました。")
            else:
                if self._does_case_21_hold(player, players):
                    types: list[CardType] = [
                        CardType.TORRENTIAL_RAIN_L,
                        CardType.THUNDERSTORM_L,
                        CardType.SHOWER_RAIN_L,
                        CardType.HEAVY_RAIN,
                        CardType.RAINY_WEATHER,
                    ]
                    other_cards = to_play_card.get_non_specified_types_cards(
                        in_hand, types
                    )
                elif self._does_case_22_hold(player, players):
                    types: list[CardType] = [
                        CardType.TORRENTIAL_RAIN_N,
                        CardType.THUNDERSTORM_N,
                        CardType.SHOWER_RAIN_N,
                    ]
                    other_cards = to_play_card.get_non_specified_types_cards(
                        in_hand, types
                    )
                else:
                    text: str = f"{player.name}の戦術： どのケースにも"
                    self._logger.info(f"{text}該当しませんでした。")
                    other_cards = []

            card: Card = random.choice(other_cards if any(other_cards) else in_hand)

        return card

    def _does_case_01_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が 11 以上のトップで、対象プレイヤーが「放流」か「日照り（全）」か
        「日照り（多）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_largest_water_amount(player, players))
        cases.append(player.water_storage >= 11)

        types: list[CardType] = [
            CardType.RELEASE,
            CardType.DROUGHT_A,
            CardType.DROUGHT_L,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 11 以上のトップで、"
            text += "対象プレイヤーが「放流」か「日照り（全）」か「日照り（多）」"
            text += "のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_02_hold(self, player: Player, players: list[Player]) -> bool:
        """全プレイヤーの貯水量が 3 または 4 で横並びのときに、対象プレイヤーが
        「日照り（全）」か「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.are_all_water_amounts_equal(players))
        cases.append((player.water_storage == 3) or (player.water_storage == 4))

        types: list[CardType] = [CardType.DROUGHT_A, CardType.DROUGHT_L]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "全プレイヤーの貯水量が 3 または 4 で横並びのときに、"
            text += "対象プレイヤーが「日照り（全）」か「日照り（多）」カードを"
            text += "持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_03_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 4 以下で、他に貯水量が 5 以上のプレイヤーが存在するとき、
        対象プレイヤーが「日照り（全）」カードを持っていれば、50% の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(human_player.water_storage <= 4)
        cases.append(any([p for p in players if p.water_storage >= 5]))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_A]))
        cases.append(random.randint(0, 1) == 0)

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 4 以下で、他に貯水量が 5 以上の"
            text += "プレイヤーが存在するとき、対象プレイヤーが「日照り（全）」"
            text += "カードを持っており、50% の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_04_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、対象プレイヤーの次の順番が人間プレイヤーで、その貯水量が
        3 以上のとき、対象プレイヤーが「集中豪雨（次）」か「雷雨（次）」か「にわか雨（次）」の
        いずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))
        cases.append(player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage >= 3)  # type: ignore

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_N,
            CardType.THUNDERSTORM_N,
            CardType.SHOWER_RAIN_N,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの序盤から中盤において、対象プレイヤーの次の順番が"
            text += "人間プレイヤーで、その貯水量が 3 以上のとき、対象プレイヤー"
            text += "が「集中豪雨（次）」か「雷雨（次）」か「にわか雨（次）」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_05_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、人間プレイヤーの貯水量がトップで、他に横並びでないプレイヤーが
        ひとりでも存在するときに、対象プレイヤーが「集中豪雨（多）」か「雷雨（多）」か「にわか雨（多）」か
        「大雨（全）」か「雨（全）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))
        cases.append(self._ct.is_largest_water_amount(human_player, players))

        second = self._ct.get_2nd_largest_water_amount(players)
        cases.append(human_player.water_storage > second)

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_L,
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
            CardType.HEAVY_RAIN,
            CardType.RAINY_WEATHER,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの序盤から中盤において、人間プレイヤーの貯水量が"
            text += "トップで、他に横並びでないプレイヤーがひとりでも存在する"
            text += "ときに、対象プレイヤーが「集中豪雨（多）」か「雷雨（多）」"
            text += "か「にわか雨（多）」か「大雨（全）」か「雨（全）」のいずれか"
            text += "のカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_06_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、貯水量が 11 以上でトップの人間以外のプレイヤーが存在し、
        他に貯水量が 2 以下の人間以外のプレイヤーが存在せず、かつ、対象プレイヤーが
        「日照り（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))

        largest = self._ct.get_largest_water_amount(players)
        cases.append(largest >= 11)
        cases.append(
            any(
                [
                    p
                    for p in players
                    if ((p.water_storage == largest) and (not p.is_human))
                ]
            )
        )
        cases.append(
            not any([p for p in players if (p.water_storage <= 2) and (not p.is_human)])
        )
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_A]))

        if all(cases):
            text: str = "ゲームの序盤から中盤において、貯水量が 11 以上でトップの"
            text += "人間以外のプレイヤーが存在し、他に貯水量が 2 以下の人間以外の"
            text += "プレイヤーが存在せず、かつ、対象プレイヤーが「日照り（全）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_07_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、貯水量が 11 以上でトップの人間以外のプレイヤーが存在し、
        対象プレイヤーが「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))

        largest = self._ct.get_largest_water_amount(players)
        cases.append(largest >= 11)
        cases.append(
            any(
                [
                    p
                    for p in players
                    if ((p.water_storage == largest) and (not p.is_human))
                ]
            )
        )
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "ゲームの序盤から中盤において、貯水量が 11 以上でトップの"
            text += "人間以外のプレイヤーが存在し、対象プレイヤーが「日照り（多）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_08_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、人間プレイヤー以外でまだ失格になっていないプレイヤーの 70% 以上が
        貯水量が 2 以下で最少のとき、対象プレイヤーが「台風（少）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))

        lowests: list[Player] = self._ct.get_players_of_lowest_water_amount(players)
        cases.append(human_player not in lowests)
        cases.append(lowests[0].water_storage <= 2)
        cases.append(len(lowests) >= math.floor(len(players) * 0.7))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TYPHOON]))

        if all(cases):
            text: str = "ゲームの序盤から中盤において、人間プレイヤー以外でまだ"
            text += "失格になっていないプレイヤーの 70% 以上が貯水量が 2 以下で"
            text += "最少のとき、対象プレイヤーが「台風（少）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_09_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、人間プレイヤー以外のプレイヤーの貯水量が 8 以下で最少のときに、
        対象プレイヤーが「台風（少）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(
            human_player not in self._ct.get_players_of_lowest_water_amount(players)
        )
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TYPHOON]))

        if all(cases):
            text: str = "ゲームの終盤において、人間プレイヤー以外のプレイヤーの"
            text += "貯水量が 8 以下で最少のときに、対象プレイヤーが「台風（少）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_10_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、人間プレイヤーの貯水量がトップのときに、対象プレイヤーが
        「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(self._ct.is_largest_water_amount(human_player, players))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "ゲームの終盤において、人間プレイヤーの貯水量がトップの"
            text += "ときに、対象プレイヤーが「日照り（多）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_11_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの貯水量が 10 以下で、
        「集中豪雨（自）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(player.water_storage <= 10)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_S]))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの貯水量が"
            text += " 10 以下で、「集中豪雨（自）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_12_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの次の順番が人間以外のプレイヤーで、
        その貯水量が 10 以下のとき、対象プレイヤーが「にわか雨（次）」か「雷雨（次）」か
        「集中豪雨（次）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(not player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage <= 10)  # type: ignore

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_N,
            CardType.THUNDERSTORM_N,
            CardType.SHOWER_RAIN_N,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの次の順番が"
            text += "人間以外のプレイヤーで、その貯水量が 10 以下のとき、"
            text += "対象プレイヤーが「にわか雨（次）」か「雷雨（次）」か"
            text += "「集中豪雨（次）」のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_13_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの次の順番が人間以外のプレイヤーで、
        その貯水量が 12 以下のとき、対象プレイヤーが「にわか雨（次）」か「雷雨（次）」の
        いずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(not player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage <= 12)  # type: ignore

        types: list[CardType] = [
            CardType.THUNDERSTORM_N,
            CardType.SHOWER_RAIN_N,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの次の順番が人間以外の"
            text += "プレイヤーで、その貯水量が 12 以下のとき、対象プレイヤーが"
            text += "「にわか雨（次）」か「雷雨（次）」のいずれかの"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_14_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの次の順番が人間以外のプレイヤーで、
        その貯水量が 14 以下のとき、対象プレイヤーが「にわか雨（次）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(not player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage <= 14)  # type: ignore
        cases.append(self._ct.has_specified_card(in_hand, [CardType.SHOWER_RAIN_N]))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの次の順番が人間以外の"
            text += "プレイヤーで、その貯水量が 14 以下のとき、対象プレイヤーが"
            text += "「にわか雨（次）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_15_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、人間以外のプレイヤーがトップで、その貯水量が 10 以下のとき、
        対象プレイヤーが「にわか雨（多）」か「雷雨（多）」か「集中豪雨（多）」のいずれかの
        カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(human_player.water_storage < largest)
        cases.append(largest <= 10)

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_L,
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの終盤において、人間以外のプレイヤーがトップで、"
            text += "その貯水量が 10 以下のとき、対象プレイヤーが"
            text += "「にわか雨（多）」か「雷雨（多）」か「集中豪雨（多）」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_16_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、人間以外のプレイヤーがトップで、その貯水量が 12 以下のとき、
        対象プレイヤーが「にわか雨（多）」か「雷雨（多）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(human_player.water_storage < largest)
        cases.append(largest <= 12)

        types: list[CardType] = [
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "ゲームの終盤において、人間以外のプレイヤーがトップで、"
            text += "その貯水量が 12 以下のとき、対象プレイヤーが「にわか雨（多）」"
            text += "か「雷雨（多）」のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_17_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、人間以外のプレイヤーがトップで、その貯水量が 14 以下のとき、
        対象プレイヤーが「にわか雨（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(human_player.water_storage < largest)
        cases.append(largest <= 14)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.SHOWER_RAIN_L]))

        if all(cases):
            text: str = "ゲームの終盤において、人間以外のプレイヤーがトップで、"
            text += "その貯水量が 14 以下のとき、対象プレイヤーが「にわか雨（多）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_21_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、人間プレイヤーがトップでなければ True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player = [p for p in players if p.is_human][0]

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))

        largest = self._ct.get_largest_water_amount(players)
        cases.append(human_player.water_storage < largest)

        if all(cases):
            text: str = "ゲームの序盤から中盤において、"
            text += "人間プレイヤーがトップでない"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_22_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの序盤から中盤において、対象プレイヤーの次の順番が人間プレイヤーでなければ True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(not self._ct.is_late_stage(players))
        cases.append(not player.next_player.is_human)  # type: ignore

        if all(cases):
            text: str = "ゲームの序盤から中盤において、"
            text += "対象プレイヤーの次の順番が人間プレイヤーでない"
            self._ct.output_case_to_log(player, text)

        return all(cases)
