"""
２人プレイ時のコンピューターの戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import random

import constants
import controllers.tactics.to_exchange_cards as to_exchange_cards
import controllers.tactics.to_play_card as to_play_card
import models.situation_in_hand as situation_in_hand
from controllers.tactics.common_tactics import CommonTactics
from models.card import Card
from models.card_type import CardType
from models.player import Player


class In2PlayersMode:
    """２人プレイ時のコンピューターの戦術。"""

    __slots__ = ["_logger", "_ct"]

    def __init__(self, logger: logging.Logger) -> None:
        """コンストラクタ。

        Args:
            logger (logging.Logger): logging.Logger のインスタンス。
        """

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

        self._ct: CommonTactics = CommonTactics(logger)
        """各モード共通のコンピューターの戦術"""

    def select_cards_to_exchange(self, player: Player) -> list[Card]:
        """「２人プレイ」モードのときに、交換する手札を選択して、その手札の list を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            list[Card]: 手札の list。
        """

        cards: list[Card] = player.cards_in_hand

        types: dict[CardType, int] = {
            CardType.RELEASE: 2,
            CardType.TORRENTIAL_RAIN_S: 1,
            CardType.DRAW_OWN_2_CARDS: 1,
        }

        return to_exchange_cards.leave_target_cards(cards, types)

    def select_card_to_play(self, player: Player, players: list[Player]) -> Card:
        """「２人プレイ」モードのときに、場に出す手札を選択して返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            Card: 場に出すカード。
        """

        in_hand: list[Card] = player.cards_in_hand

        if (c := self._ct.get_card_in_common_cases(player, players)) is not None:
            card: Card = c
        elif self._does_case_01_hold(player, players):
            types: list[CardType] = [
                CardType.RELEASE,
                CardType.DROUGHT_A,
                CardType.DROUGHT_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_02_hold(player):
            types: list[CardType] = [
                CardType.HEAVY_RAIN,
                CardType.RAINY_WEATHER,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_03_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_S]
            )
        elif self._does_case_04_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_05_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_06_hold(player, players):
            types: list[CardType] = [
                CardType.DROUGHT_L,
                CardType.DROUGHT_A,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_07_hold(player, players):
            types: list[CardType] = [
                CardType.DROUGHT_L,
                CardType.DROUGHT_A,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_08_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.SHOWER_RAIN_N]
            )
        elif self._does_case_09_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.SHOWER_RAIN_N]
            )
        elif self._does_case_10_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.RELEASE])
        elif self._does_case_11_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.RELEASE])
        elif self._does_case_12_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_13_hold(player, players):
            types: list[CardType] = [
                CardType.TORRENTIAL_RAIN_L,
                CardType.THUNDERSTORM_L,
                CardType.SHOWER_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_14_hold(player, players):
            types: list[CardType] = [
                CardType.THUNDERSTORM_L,
                CardType.SHOWER_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_15_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.SHOWER_RAIN_L]
            )
        else:
            water_storage: int = player.water_storage
            is_largest: bool = self._ct.is_largest_water_amount(player, players)
            other_cards: list[Card] = to_play_card.get_cards_in_other_cases(
                in_hand, water_storage, is_largest
            )

            text: str = f"{player.name}の戦術："
            if any(other_cards):
                self._logger.info(f"{text}「その他のケース」に該当しました。")
            else:
                self._logger.info(f"{text} どのケースにも該当しませんでした。")

            card: Card = random.choice(other_cards if any(other_cards) else in_hand)

        return card

    def _does_case_01_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量がトップで、貯水量がしきい値を超えており、かつ、「放流」か
        「日照り（全）」か「日照り（多）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(self._ct.is_largest_water_amount(player, players))
        cases.append(player.water_storage >= self._get_threshold(9, 2.0))

        types: list[CardType] = [
            CardType.RELEASE,
            CardType.DROUGHT_A,
            CardType.DROUGHT_L,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量がトップで、貯水量がしきい値を"
            text += "超えており、かつ、「放流」か「日照り（全）」か「日照り（多）」"
            text += "のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_02_hold(self, player: Player) -> bool:
        """対象プレイヤーの貯水量が 2 以下で、「雨」か「大雨」の
        いずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.water_storage <= 2)

        types: list[CardType] = [
            CardType.HEAVY_RAIN,
            CardType.RAINY_WEATHER,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 2 以下で、「雨」か「大雨」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_03_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が 2 以下で、「集中豪雨（自）」カードを持っているときに、
        50% の確率でフラグが立つか、または、ゲームの終盤であれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.water_storage <= 2)

        types: list[CardType] = [CardType.TORRENTIAL_RAIN_S]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        cases.append(self._ct.is_late_stage(players) or (random.randint(0, 1) == 0))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 2 以下で、「集中豪雨（自）」"
            text += "カードを持っているときに、50% の確率でフラグが立つか、"
            text += "または、ゲームの終盤である"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_04_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が 2 以下で、「集中豪雨（自）」カードは持っておらず、
        「台風（少）」カードを持っていれば、75% の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.water_storage <= 2)

        types: list[CardType] = [CardType.TORRENTIAL_RAIN_S]
        cases.append(not self._ct.has_specified_card(player.cards_in_hand, types))
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.TYPHOON])
        )
        cases.append(self._ct.is_late_stage(players) or (random.randint(0, 3) > 0))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 2 以下で、「集中豪雨（自）」"
            text += "カードは持っておらず、「台風（少）」カードを持っており、"
            text += "75% の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_05_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲーム終盤、対象プレイヤーの貯水量が 8 以下で、人間プレイヤーの貯水量を下回っており、
        かつ、「台風（少）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(player.water_storage <= 8)
        cases.append(player.water_storage < human_player.water_storage)
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.TYPHOON])
        )

        if all(cases):
            text: str = "ゲーム終盤、対象プレイヤーの貯水量が 8 以下で、"
            text += "人間プレイヤーの貯水量を下回っており、かつ、"
            text += "「台風（少）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_06_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、どちらも 4 以下であり、対象プレイヤーが「日照り（全）」か
        「日照り（多）」カードを合わせて 2 枚以上と、「集中豪雨（自）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(human_player.water_storage == player.water_storage)
        cases.append(player.water_storage <= 4)

        types: list[CardType] = [CardType.DROUGHT_A, CardType.DROUGHT_L]
        cases.append(situation_in_hand.has_specific_cards_more_than(in_hand, types, 2))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_S]))

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "どちらも 4 以下であり、対象プレイヤーが「日照り（全）」か"
            text += "「日照り（多）」カードを合わせて 2 枚以上と、"
            text += "「集中豪雨（自）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_07_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、どちらも 4 以下であり、
        かつ、対象プレイヤーが「日照り（全）」か「日照り（多）」カードを合わせて
        2 枚以上持っていれば、1/3 の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(human_player.water_storage == player.water_storage)
        cases.append(player.water_storage <= 4)

        types: list[CardType] = [CardType.DROUGHT_A, CardType.DROUGHT_L]
        cases.append(situation_in_hand.has_specific_cards_more_than(in_hand, types, 2))
        cases.append(random.randint(0, 2) == 0)

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "どちらも 4 以下であり、かつ、対象プレイヤーが「日照り（全）」"
            text += "か「日照り（多）」カードを合わせて 2 枚以上持っており、"
            text += "1/3 の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_08_hold(
        self,
        player: Player,
        players: list[Player],
    ) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、対象プレイヤーが「にわか雨（次）」と
        「日照り（多）」カードを持っていれば、2/3 の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage == human_player.water_storage)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.SHOWER_RAIN_N]))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))
        cases.append(random.randint(0, 2) != 0)

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "対象プレイヤーが「にわか雨（次）」と「日照り（多）」カード"
            text += "を持っており、2/3 の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_09_hold(
        self,
        player: Player,
        players: list[Player],
    ) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、対象プレイヤーが
        「にわか雨（次）」カードを持っていれば、1/3 の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage == human_player.water_storage)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.SHOWER_RAIN_N]))
        cases.append(random.randint(0, 2) == 0)

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "対象プレイヤーが「にわか雨（次）」カードを持っており、"
            text += "1/3 の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_10_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、どちらも 4 以上であり、かつ、対象プレイヤーが
        「放流」と「日照り（多）」カードを持っていれば、2/3 の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage >= 4)
        cases.append(player.water_storage == human_player.water_storage)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.RELEASE]))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))
        cases.append(random.randint(0, 2) != 0)

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "どちらも 4 以上であり、かつ、対象プレイヤーが「放流」と"
            text += "「日照り（多）」カードを持っており、"
            text += "2/3 の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_11_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーと人間プレイヤーの貯水量が同じで、どちらも 4 以上であり、
        対象プレイヤーが「放流」カードを持っていれば、1/3 の確率で True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage >= 4)
        cases.append(player.water_storage == human_player.water_storage)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.RELEASE]))
        cases.append(random.randint(0, 2) == 0)

        if all(cases):
            text: str = "対象プレイヤーと人間プレイヤーの貯水量が同じで、"
            text += "どちらも 4 以上であり、対象プレイヤーが「放流」カードを"
            text += "持っており、1/3 の確率でフラグが立った"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_12_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が対象プレイヤーの貯水量を 1 上回っており、対象プレイヤーが
        「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(human_player.water_storage - player.water_storage == 1)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が対象プレイヤーの貯水量を"
            text += " 1 上回っており、対象プレイヤーが「日照り（多）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_13_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が人間プレイヤーの貯水量を 1 上回っており、対象プレイヤーの
        貯水量が 3 前後のしきい値以下のとき、対象プレイヤーが「集中豪雨（多）」か
        「雷雨（多）」か「にわか雨（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage - human_player.water_storage == 1)
        cases.append(player.water_storage <= self._get_threshold(3, 0.8))

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_L,
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が人間プレイヤーの貯水量を"
            text += " 1 上回っており、対象プレイヤーの貯水量が 3 前後のしきい値"
            text += "以下のとき、対象プレイヤーが「集中豪雨（多）」か「雷雨（多）」"
            text += "か「にわか雨（多）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_14_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が人間プレイヤーの貯水量を 1 上回っており、対象プレイヤーの
        貯水量が 5 前後のしきい値以下のとき、対象プレイヤーが「雷雨（多）」か「にわか雨（多）」
        カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage - human_player.water_storage == 1)
        cases.append(player.water_storage <= self._get_threshold(5, 1.0))

        types: list[CardType] = [
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
        ]
        cases.append(self._ct.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が人間プレイヤーの貯水量を"
            text += " 1 上回っており、対象プレイヤーの貯水量が 5 前後のしきい値"
            text += "以下のとき、対象プレイヤーが「雷雨（多）」か「にわか雨（多）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_15_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が人間プレイヤーの貯水量を 1 上回っており、対象プレイヤーの
        貯水量が 7 前後のしきい値以下のとき、対象プレイヤーが「にわか雨（多）」カードを
        持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.water_storage - human_player.water_storage == 1)
        cases.append(player.water_storage <= self._get_threshold(7, 1.2))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.SHOWER_RAIN_L]))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が人間プレイヤーの貯水量を"
            text += " 1 上回っており、対象プレイヤーの貯水量が 7 前後のしきい値"
            text += "以下のとき、対象プレイヤーが「にわか雨（多）」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _get_threshold(self, standard_value: int, sigma: float) -> int:
        """貯水量を増減させる際に基準となるしきい値を、指定された基準値を
        中心にランダムに算出して返す。

        Args:
            standard_value (int): 基準値。
            sigma (float): 基準値に対する標準偏差。

        Returns:
            int: ランダムに算出された整数値。
        """

        offset: int = int(random.normalvariate(0, sigma))
        threshold: int = standard_value + offset

        LOWER: int = 2
        UPPER: int = constants.WATER_CAPACITY

        return (
            LOWER
            if (threshold < LOWER)
            else (UPPER if (threshold > UPPER) else threshold)
        )
