"""
各モード共通のコンピューターの戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import random

import controllers.tactics.to_play_card as to_play_card
from models.card import Card
from models.card_type import CardType
from models.player import Player


class CommonTactics:
    """各モード共通のコンピューターの戦術。"""

    __slots__ = ["_logger"]

    def __init__(self, logger: logging.Logger) -> None:
        """各モード共通のコンピューターの戦術。

        Args:
            logger (logging.Logger): logging.Logger のインスタンス。
        """

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

    def get_card_in_common_cases(
        self, player: Player, players: list[Player]
    ) -> Card | None:
        """「２人プレイ」と「ちょいムズ」モード共通の「人間プレイヤーを失格にできる」ケースにおいて、
        手札の中から最適なカードを選択して返す。該当がない場合は None を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            Card | None: 選択されたカード。該当がない場合は None。
        """

        in_hand: list[Card] = player.cards_in_hand

        if self._does_case_01_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_02_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_03_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_04_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_05_hold(player):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_N,
                CardType.THUNDERSTORM_N,
                CardType.TORRENTIAL_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_06_hold(player, players):
            types: list[CardType] = [
                CardType.RAINY_WEATHER,
                CardType.SHOWER_RAIN_L,
                CardType.THUNDERSTORM_L,
                CardType.TORRENTIAL_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_07_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.RAINY_WEATHER]
            )
        elif self._does_case_08_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.HEAVY_RAIN]
            )
        elif self._does_case_09_hold(player):
            types: list[CardType] = [
                CardType.THUNDERSTORM_N,
                CardType.TORRENTIAL_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_10_hold(player, players):
            types: list[CardType] = [
                CardType.THUNDERSTORM_L,
                CardType.TORRENTIAL_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_11_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.HEAVY_RAIN]
            )
        elif self._does_case_12_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.HEAVY_RAIN]
            )
        elif self._does_case_13_hold(player):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_N]
            )
        elif self._does_case_14_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_L]
            )
        elif self._does_case_15_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_16_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        else:
            return None

        return card

    def _does_case_01_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 2 以下のとき、他に貯水量が 3 以上のプレイヤーが存在し、
        かつ、対象プレイヤーが「日照り（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        in_hand: list[Card] = player.cards_in_hand

        cases.append([p for p in players if p.is_human][0].water_storage <= 2)
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage >= 3)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.DROUGHT_A]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 2 以下のとき、他に貯水量が"
            text += " 3 以上のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「日照り（全）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_02_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 2 以下のとき、対象プレイヤーが「集中豪雨（自）」と
        「日照り（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        in_hand: list[Card] = player.cards_in_hand

        cases.append([p for p in players if p.is_human][0].water_storage <= 2)
        cases.append(self.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_S]))
        cases.append(self.has_specified_card(in_hand, [CardType.DROUGHT_A]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 2 以下のとき、対象プレイヤーが"
            text += "「集中豪雨（自）」と「日照り（全）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_03_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 2 でトップのとき、他に貯水量が 1 のプレイヤーが存在し、
        かつ、対象プレイヤーが「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage == 2)
        cases.append(self.is_largest_water_amount(human_player, players))
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage == 1)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 2 でトップのとき、"
            text += "他に貯水量が 1 のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「日照り（多）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_04_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 2 でトップのとき、対象プレイヤーが「集中豪雨（自）」と
        「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage == 2)
        cases.append(self.is_largest_water_amount(human_player, players))
        cases.append(self.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_S]))
        cases.append(self.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 2 でトップのとき、対象プレイヤー"
            text += "が「集中豪雨（自）」と「日照り（多）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_05_hold(self, player: Player) -> bool:
        """次の順番が人間プレイヤーで、その貯水量が 15 のとき、対象プレイヤーが「にわか雨（次）」か
        「雷雨（次）」か「集中豪雨（次）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []

        cases.append(player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage == 15)  # type: ignore

        types: list[CardType] = [
            CardType.SHOWER_RAIN_N,
            CardType.THUNDERSTORM_N,
            CardType.TORRENTIAL_RAIN_N,
        ]
        cases.append(self.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "次の順番が人間プレイヤーで、その貯水量が 15 のとき、"
            text += "対象プレイヤーが「にわか雨（次）」か「雷雨（次）」か"
            text += "「集中豪雨（次）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_06_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 15 で、他に貯水量が 14 以下のプレイヤーが存在し、
        かつ、対象プレイヤーが「にわか雨（多）」か「雷雨（多）」か「集中豪雨（多）」か
        「雨（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage == 15)
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage <= 14)])
        )

        types: list[CardType] = [
            CardType.SHOWER_RAIN_L,
            CardType.THUNDERSTORM_L,
            CardType.TORRENTIAL_RAIN_L,
            CardType.RAINY_WEATHER,
        ]
        cases.append(self.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 15 で、他に貯水量が"
            text += " 14 以下のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「にわか雨（多）」か「雷雨（多）」か「集中豪雨（多）」か"
            text += "「雨（全）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_07_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 15 のとき、対象プレイヤーが「雨（全）」と
        「放流」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage == 15)
        cases.append(self.has_specified_card(in_hand, [CardType.RAINY_WEATHER]))
        cases.append(self.has_specified_card(in_hand, [CardType.RELEASE]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 15 のとき、対象プレイヤーが"
            text += "「雨（全）」と「放流」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_08_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 15 で、他に貯水量が 12 以下のプレイヤーが存在し、
        かつ、対象プレイヤーが「大雨（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage == 15)
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage <= 12)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.HEAVY_RAIN]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 15 で、他に貯水量が"
            text += " 12 以下のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「大雨（全）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_09_hold(self, player: Player) -> bool:
        """次の順番のプレイヤーが人間で、その貯水量が 13 以上のとき、対象プレイヤーが
        「雷雨（次）」か「集中豪雨（次）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        in_hand = player.cards_in_hand

        cases.append(player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage >= 13)  # type: ignore

        types: list[CardType] = [
            CardType.THUNDERSTORM_N,
            CardType.TORRENTIAL_RAIN_N,
        ]
        cases.append(self.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "次の順番のプレイヤーが人間で、その貯水量が"
            text += " 13 以上のとき、対象プレイヤーが「雷雨（次）」か"
            text += "「集中豪雨（次）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_10_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 13 以上のトップで、他に貯水量が 12 以下のプレイヤーが存在し、
        かつ、対象プレイヤーが「雷雨（多）」か「集中豪雨（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 13)
        cases.append(self.is_largest_water_amount(human_player, players))
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage <= 12)])
        )

        types: list[CardType] = [
            CardType.THUNDERSTORM_L,
            CardType.TORRENTIAL_RAIN_L,
        ]
        cases.append(self.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 13 以上のトップで、他に貯水量が"
            text += " 12 以下のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「雷雨（多）」か「集中豪雨（多）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_11_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 13 以上で、他に貯水量が 12 以下のプレイヤーが存在し、
        かつ、対象プレイヤーが「大雨（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 13)
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage <= 12)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.HEAVY_RAIN]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 13 以上で、他に貯水量が"
            text += " 12 以下のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「大雨（全）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_12_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 13 以上のとき、対象プレイヤーが「大雨（全）」と
        「放流」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 13)
        cases.append(self.has_specified_card(in_hand, [CardType.HEAVY_RAIN]))
        cases.append(self.has_specified_card(in_hand, [CardType.RELEASE]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 13 以上のとき、対象プレイヤー"
            text += "が「大雨（全）」と「放流」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_13_hold(self, player: Player) -> bool:
        """次の順番のプレイヤーが人間で、その貯水量が 11 以上のとき、対象プレイヤーが
        「集中豪雨（次）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        in_hand: list[Card] = player.cards_in_hand

        cases.append(player.next_player.is_human)  # type: ignore
        cases.append(player.next_player.water_storage >= 13)  # type: ignore

        types: list[CardType] = [CardType.TORRENTIAL_RAIN_N]
        cases.append(self.has_specified_card(in_hand, types))

        if all(cases):
            text: str = "次の順番のプレイヤーが人間で、その貯水量が 11 以上のとき、"
            text += "対象プレイヤーが「集中豪雨（次）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_14_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 11 以上のトップで、他に貯水量が 10 以下のプレイヤーが存在し、
        かつ、対象プレイヤーが「集中豪雨（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 11)
        cases.append(self.is_largest_water_amount(human_player, players))
        cases.append(
            any([p for p in players if ((not p.is_human) and p.water_storage <= 10)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_L]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 11 以上のトップで、他に貯水量が"
            text += " 10 以下のプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「集中豪雨（多）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_15_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 9 以上の最少で、他に貯水量が最少ではないプレイヤーが存在し、
        かつ、対象プレイヤーが「台風（少）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 9)
        cases.append(self.is_lowest_water_amount(human_player, players))
        cases.append(
            any([p for p in players if (p.water_storage > human_player.water_storage)])
        )
        cases.append(self.has_specified_card(in_hand, [CardType.TYPHOON]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 9 以上の最少で、他に貯水量が"
            text += "最少ではないプレイヤーが存在し、かつ、対象プレイヤーが"
            text += "「台風（少）」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def _does_case_16_hold(self, player: Player, players: list[Player]) -> bool:
        """人間プレイヤーの貯水量が 9 以上で最少のとき、
        対象プレイヤーが「台風（少）」と「放流」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        human_player: Player = [p for p in players if p.is_human][0]
        in_hand: list[Card] = player.cards_in_hand

        cases.append(human_player.water_storage >= 9)
        cases.append(self.is_lowest_water_amount(human_player, players))
        cases.append(self.has_specified_card(in_hand, [CardType.TYPHOON]))
        cases.append(self.has_specified_card(in_hand, [CardType.RELEASE]))

        if all(cases):
            text: str = "人間プレイヤーの貯水量が 9 以上で最少のとき、"
            text += "対象プレイヤーが「台風（少）」と「放流」カードを持っている"
            self.output_case_to_log(player, text)

        return all(cases)

    def has_specified_card(self, cards: list[Card], card_types: list[CardType]) -> bool:
        """カードの list の中に、指定された種類のカードが存在すれば True を返す。

        Args:
            cards (list[Card]): カードの list。
            card_types (list[CardType]): 存在を確認するカードの種類。

        Returns:
            bool: カードの list の中に、指定された種類のカードが存在すれば True。
        """

        return any([c for c in cards if (c.type in card_types)])

    def is_late_stage(self, players: list[Player]) -> bool:
        """各プレイヤーの残りの手札の数から、ゲームの終盤と見なす場合は True を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: ゲームの終盤と見なす場合は True。
        """

        if len(players) < 2:
            return False
        elif len(players) == 2:
            threshold: int = random.randint(4, 5)
            return any([p for p in players if len(p.cards_in_hand) <= threshold])
        else:
            threshold: int = 2 if (len(players) < 5) else random.randint(1, 2)
            return any([p for p in players if len(p.cards_in_hand) <= threshold])

    def is_largest_water_amount(self, player: Player, players: list[Player]) -> bool:
        """指定されたプレイヤーの貯水量が、失格になっていないプレイヤーの中で最大であれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 貯水量が最大であれば True。
        """

        return player in self.get_players_of_largest_water_amount(players)

    def is_2nd_largest_water_amount(
        self, player: Player, players: list[Player]
    ) -> bool:
        """指定されたプレイヤーの貯水量が、失格になっていないプレイヤーの中で２番目であれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 貯水量が２番目であれば True。
        """

        second_amount: int = self.get_2nd_largest_water_amount(players)
        return player.water_storage == second_amount

    def is_lowest_water_amount(self, player: Player, players: list[Player]) -> bool:
        """指定されたプレイヤーの貯水量が、失格になっていないプレイヤーの中で最少であれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 貯水量が最少であれば True。
        """

        return player in self.get_players_of_lowest_water_amount(players)

    def get_players_of_largest_water_amount(
        self, players: list[Player]
    ) -> list[Player]:
        """失格になっていないプレイヤーのうち、貯水量が最も多いプレイヤーの list を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            list[Player]: 貯水量が最大のプレイヤーの list。
        """

        largest_amount: int = max([p.water_storage for p in players])
        return [p for p in players if p.water_storage == largest_amount]

    def get_players_of_lowest_water_amount(self, players: list[Player]) -> list[Player]:
        """失格になっていないプレイヤーのうち、貯水量が最も少ないプレイヤーの list を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            list[Player]: 貯水量が最少のプレイヤーの list。
        """

        lowest_amount: int = min([p.water_storage for p in players])
        return [p for p in players if p.water_storage == lowest_amount]

    def get_largest_water_amount(self, players: list[Player]) -> int:
        """失格になっていないプレイヤーのうち、貯水量が最も多いプレイヤーの貯水量の値を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            int: 貯水量が最大のプレイヤーの貯水量。
        """

        return max([p.water_storage for p in players])

    def get_2nd_largest_water_amount(self, players: list[Player]) -> int:
        """失格になっていないプレイヤーのうち、貯水量が２番目に多いプレイヤーの貯水量の値を返す。
        該当がない場合は、貯水量が最も多いプレイヤーの値を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            int: 貯水量が２番目に多いプレイヤーの貯水量。
        """

        largest: int = max([p.water_storage for p in players])
        target: list[int] = [
            p.water_storage for p in players if p.water_storage < largest
        ]
        return max(target) if any(target) else largest

    def are_all_water_amounts_equal(self, players: list[Player]) -> bool:
        """失格になっていないプレイヤーの貯水量がすべて同じであれば True を返す。

        Args:
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: すべてのプレイヤーの貯水量が同じであれば True。
        """

        largest: int = max([p.water_storage for p in players])
        return all([(p.water_storage == largest) for p in players])

    def output_case_to_log(self, player: Player, case: str) -> None:
        """現在の状況に該当するケースをログに出力する。

        Args:
            player (Player): 対象プレイヤー。
            text (str): 該当するケース。
        """

        text: str = f"{player.name}の戦術：「{case}」ケースに該当しました。"
        self._logger.info(text)
