"""
ゲームの進行を管理する。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import random
from collections import deque
from typing import Callable

from kivy.metrics import dp

import constants
import controllers.tactics.tactics as tactics
from models.affected_player import AffectedPlayer
from models.affected_target import AffectedTarget
from models.card import Card
from models.card_type import CardType
from models.difficulty import Difficulty
from models.game_scene import GameScene
from models.my_config import MyConfig
from models.player import Player
from models.shared_vars import SharedVars
from models.sound_scene import SoundScene
from views.custom_image import CustomImage
from views.player_status import PlayerStatus


class GameManager:
    """ゲームの進行を管理するクラス。"""

    __slots__ = [
        "_config",
        "_vars",
        "_logger",
        "_root_path",
        "_add_status",
        "deck",
        "players",
        "human_player",
        "current_player",
        "cards_on_table",
        "is_clockwise",
    ]

    def __init__(
        self,
        config: MyConfig | None = None,
        shared_vars: SharedVars | None = None,
        logger: logging.Logger | None = None,
        add_status: Callable[[str], None] | None = None,
    ):
        """ゲームの進行を管理するクラスのインスタンスを作成して返す。

        Args:
            config (MyConfig, optional): アプリの設定値を管理するクラスのインスタンス。
            shared_vars (SharedVars, optional): アプリ内で共有する変数を管理するクラスのインスタンス。
            logger (logging.Logger, optional): logging.Logger のインスタンス。
            add_status (Callable[[str], None], optional): ゲーム状況の表示に 1 行分のテキストを追加する関数。
        """

        if config and shared_vars and logger and add_status:
            self._config: MyConfig = config
            """アプリの設定値を管理するクラスのインスタンス"""

            self._vars: SharedVars = shared_vars
            """アプリ内で共有する変数を管理するクラスのインスタンス"""

            self._logger: logging.Logger = logger
            """logging.Logger のインスタンス"""

            self._add_status: Callable[[str], None] = add_status
            """ゲーム状況の表示に 1 行分のテキストを追加する関数"""

            self.deck: deque[Card] = deque(self.create_cards())
            """山札の deque"""

            self.players: list[Player] = self.create_players()
            """すべてのプレイヤーの list"""

            self.human_player: Player = [p for p in self.players if p.is_human][0]
            """人間プレイヤーのインスタンス"""

            self.current_player: Player = random.choice(self.players)
            """現在の順番のプレイヤー"""

            self.cards_on_table: list[Card] = []
            """場に出されたカードの list"""

            self.is_clockwise: bool = True
            """カードを出す順番が時計回りであれば True"""

            self._deal_cards_to_all_players()

    def create_cards(self) -> list[Card]:
        """すべてのカードのインスタンスを作成し、シャッフルして list に格納して返す。

        Returns:
            list[Card]: すべてのカードのインスタンスが格納された list。
        """

        cards: list[Card] = []
        index: int = 0
        for type in CardType:
            is_card_added: bool = self._config.add_option_cards
            if (not is_card_added) and (type is CardType.TYPHOON):
                continue

            for _ in range(type.count):
                name: str = type.card_name
                source: str = type.image_filename
                size = (dp(60), dp(90))
                widget: CustomImage = CustomImage(
                    source=source, name=name, size=size, index=index
                )
                amount: int = type.water_increase_amount
                card = Card(
                    type=type,
                    index=index,
                    widget=widget,
                    name=name,
                    affected_player=type.affected_player,
                    affected_target=type.affected_target,
                    water_increase_amount=amount,
                    draw_count=type.draw_count,
                    description=type.description,
                    is_selected=False,
                )
                cards.append(card)
                index += 1

        random.shuffle(cards)

        return cards

    def create_players(self) -> list[Player]:
        """すべてのプレイヤーのインスタンスを作成し、list に格納して返す。

        Returns:
            list[Player]: すべてのプレイヤーのインスタンスが格納された list。
        """

        players: list[Player] = []

        if self._config.player_count == 2:
            names: list[str] = ["Ａさん", "あなた"]
            pos_hints: list[dict[str, float]] = self._get_pos_hints(2)
        elif self._config.player_count == 3:
            names: list[str] = ["Ａさん", "Ｂさん", "あなた"]
            pos_hints: list[dict[str, float]] = self._get_pos_hints(3)
        else:
            names: list[str] = [
                "Ａさん",
                "Ｂさん",
                "Ｃさん",
                "あなた",
                "Ｄさん",
                "Ｅさん",
            ]
            pos_hints: list[dict[str, float]] = self._get_pos_hints(6)

        for i in range(self._config.player_count):
            player: Player = Player(
                name=names[i],
                index=i,
                status=PlayerStatus(pos_hint=pos_hints[i]),
                base_point=pos_hints[i],
                is_human=(names[i] == "あなた"),
            )
            players.append(player)

        players = self._set_prev_and_next_player(players)

        return players

    def _get_pos_hints(self, player_count: int) -> list[dict[str, float]]:
        """各プレイヤーの位置を示す座標の list を返す。

        Args:
            player_count (int): プレイヤーの数。

        Returns:
            list[dict[str,float]]: 各プレイヤーの位置を示す座標の list。
        """

        if player_count == 2:
            return [
                {"x": 0.653, "y": 0.726},  # A
                {"x": 0.015, "y": 0.263},  # You
            ]
        elif player_count == 3:
            return [
                {"x": 0.653, "y": 0.726},  # A
                {"x": 0.653, "y": 0.577},  # B
                {"x": 0.015, "y": 0.263},  # You
            ]
        else:
            return [
                {"x": 0.653, "y": 0.726},  # A
                {"x": 0.653, "y": 0.577},  # B
                {"x": 0.653, "y": 0.426},  # C
                {"x": 0.015, "y": 0.263},  # You
                {"x": 0.015, "y": 0.426},  # D
                {"x": 0.015, "y": 0.577},  # E
            ]

    def _set_prev_and_next_player(self, players: list[Player]) -> list[Player]:
        """すべてのプレイヤーの前後のプレイヤーを設定する。

        Args:
            players (list[Player]): 設定前のすべてのプレイヤーの list。

        Returns:
            list[Player]: 設定後のすべてのプレイヤーの list。
        """

        active_player_count: int = len([p for p in players if (not p.is_drop_out)])

        for player in players:
            if not player.is_drop_out:
                prev: int = (player.index - 1) % active_player_count
                player.prev_player = [p for p in players if p.index == prev][0]
                next: int = (player.index + 1) % active_player_count
                player.next_player = [p for p in players if p.index == next][0]

        return players

    def _deal_cards_to_all_players(self) -> None:
        """ゲーム開始前にすべてのプレイヤーにカードを配る。"""

        start_player: Player = self.current_player

        for _ in range(constants.CARDS_TO_DEAL_COUNT):
            player: Player = start_player
            while True:
                player.cards_in_hand.append(self.deck.popleft())
                player: Player = player.next_player if player.next_player else player
                if player is start_player:
                    break

        for player in self.players:
            player.cards_in_hand.sort(key=lambda c: c.index)

    def exchange_hand(self, player: Player) -> int:
        """指定されたプレイヤーの手札を交換して、交換した枚数を返す。

        Args:
            player (Player): 対象となるプレイヤー。

        Returns:
            int: 交換した手札の枚数。
        """

        if not player.is_human:
            player_count: int = self._config.player_count
            is_hard_mode: bool = self._config.difficulty == Difficulty.HARD

            cards: list[Card] = tactics.select_cards_to_exchange(
                player=player,
                player_count=player_count,
                is_hard_mode=is_hard_mode,
                logger=self._logger,
            )
            player.cards_in_hand = cards

        selected_cards: list[Card] = [c for c in player.cards_in_hand if c.is_selected]
        selected_count: int = len(selected_cards)
        self.log_starting_hand(player, is_before=True)

        for card in selected_cards:
            player.cards_in_hand.remove(card)
            self.cards_on_table.append(card)
            player.cards_in_hand.append(self.deck.popleft())

        return selected_count

    def select_computer_card(self) -> Card:
        """コンピュータープレイヤーが場に出すカードを選択する。"""

        is_hard_mode: bool = self._config.difficulty == Difficulty.HARD
        return tactics.select_card_to_play(
            player=self.current_player,
            all_players=self.players,
            player_count=self._config.player_count,
            is_hard_mode=is_hard_mode,
            logger=self._logger,
        )

    def move_to_next_player(self, is_first: bool = False) -> None:
        """次のプレイヤーに順番を移す。

        Args:
            is_first (bool, optional): 初回の場合は True。
        """

        if self.current_player.next_player and (not is_first):
            self.current_player = self.current_player.next_player

        self.log_current_player()

        if self._vars.game_scene != GameScene.GAME_END:
            if self.current_player.is_human:
                self._vars.prev_scene = self._vars.game_scene
                self._vars.game_scene = GameScene.TURN_OF_HUMAN
            else:
                self._vars.prev_scene = self._vars.game_scene
                self._vars.game_scene = GameScene.TURN_OF_COMPUTER

    def return_played_cards_to_deck(self, return_all: bool) -> None:
        """場に出されたカードをシャッフルして山札に戻す。

        Args:
            return_all (bool): すべてのカードを戻す場合は True。
        """

        game_scene: GameScene = self._vars.game_scene
        applicable_scene: bool = game_scene is GameScene.EXCHANGE_CARDS
        if applicable_scene or (len(self.deck) <= 12):
            if return_all:
                card_on_table: Card = self.cards_on_table[-1]
            else:
                card_on_table: Card = self.cards_on_table.pop()

            random.shuffle(self.cards_on_table)
            self.deck.extend(self.cards_on_table)
            self.cards_on_table = [] if return_all else [card_on_table]

            self._logger.info("山札にカードを戻しました。")

    ###############################################################################
    # カードを出す処理

    def play_card(
        self, card: Card, transition_to_release_input_screen: Callable[[int], None]
    ) -> None:
        """現在の順番のプレイヤーが指定されたカードを場に出す。

        Args:
            card (Card): 場に出すカードのインスタンス。
            transition_to_release_input_screen (Callable[[int], None]): 放流量の入力画面に遷移する関数。
        """

        self.current_player.cards_in_hand.remove(card)
        self.cards_on_table.append(card)

        text: str = f"{self.current_player.name} は「{card.name}」を出しました。"
        self._display_status_and_log(text)
        self._vars.sounds[SoundScene.CARD_FLIP].play()

        if card.type is CardType.REVERSE:
            self._reverse_order()
        elif card.affected_target is AffectedTarget.WATER_STORAGE:
            self._apply_water_storage_effect(card, transition_to_release_input_screen)
            self._output_players_exceeding_range()
        elif card.affected_target is AffectedTarget.CARDS_IN_HAND:
            self._apply_cards_in_hand_effect(card)

    def _output_players_exceeding_range(self) -> None:
        """貯水量が範囲を超えたプレイヤーをログと画面に出力する。

        Args:
            players (list[Player]): 貯水量が範囲を超えたプレイヤーの list。
        """

        players: list[Player] = self.get_players_exceeding_range()
        for player in players:
            if player.water_storage <= 0:
                text: str = f"{player.name} の貯水量が 0 万㎥以下になりました。"
            else:
                limit: str = f" {constants.WATER_CAPACITY * 100} 万㎥"
                text: str = f"{player.name} の貯水量が{limit}を超えました。"

            self._display_status_and_log(text)

    def _reverse_order(self) -> None:
        """カードを出す順番を逆回りにする。"""

        self.is_clockwise = not self.is_clockwise

        self.players.reverse()

        self._set_player_indexes()
        self.players = self._set_prev_and_next_player(self.players)

        active_players: list[Player] = [p for p in self.players if (not p.is_drop_out)]
        if len(active_players) == 2:
            text: str = "効果はありませんでした。"
        else:
            order: str = "時計" if self.is_clockwise else "反時計"
            text: str = f"カードを出す順番が {order}回り になりました。"

        self._display_status_and_log(text)

    def _set_player_indexes(self) -> None:
        """すべてのプレイヤーの index を設定する。"""

        i: int = 0
        for player in self.players:
            if not player.is_drop_out:
                player.index = i
                i += 1

    def _apply_water_storage_effect(
        self, card: Card, transition_to_release_input_screen: Callable[[int], None]
    ) -> None:
        """場に出されたカードの効果を貯水量に適用する。

        Args:
            card (Card): 場に出されたカードのインスタンス。
            transition_to_release_input_screen (Callable[[int], None]): 放流量の入力画面に遷移する関数。
        """

        all_players: list[Player] = [p for p in self.players if (not p.is_drop_out)]

        if card.type is CardType.RELEASE:
            self._apply_release_effect(transition_to_release_input_screen)
        elif card.affected_player is AffectedPlayer.SELF:
            self.current_player.water_storage += card.water_increase_amount
            self._output_water_storage_effect(self.current_player.name, card)
        elif card.affected_player is AffectedPlayer.ALL_PLAYERS:
            for player in all_players:
                player.water_storage += card.water_increase_amount
            self._output_water_storage_effect("全員", card)
        elif card.affected_player is AffectedPlayer.NEXT_PLAYER:
            next_player: Player = (
                self.current_player.next_player
                if self.current_player.next_player
                else self.current_player
            )
            next_player.water_storage += card.water_increase_amount
            self._output_water_storage_effect(next_player.name, card)
        elif card.affected_player is AffectedPlayer.PLAYER_WITH_MOST_WATER:
            max_amount: int = max([p.water_storage for p in all_players])
            players: list[Player] = [
                p for p in all_players if p.water_storage == max_amount
            ]
            for player in players:
                player.water_storage += card.water_increase_amount
                self._output_water_storage_effect(player.name, card)
        elif card.affected_player is AffectedPlayer.PLAYER_WITH_LOWEST_WATER:
            min_amount: int = min([p.water_storage for p in all_players])
            players: list[Player] = [
                p for p in all_players if p.water_storage == min_amount
            ]
            for player in players:
                player.water_storage += card.water_increase_amount
                self._output_water_storage_effect(player.name, card)

    def _apply_release_effect(
        self, transition_to_release_input_screen: Callable[[int], None]
    ) -> None:
        """「放流」の効果を適用する。

        Args:
            transition_to_release_input_screen (Callable[[int], None]): 放流量の入力画面に遷移する関数。
        """

        remaining: int = tactics.get_amount_of_water_after_release(
            self.current_player, self.players, self._logger
        )
        if self.current_player.is_human:
            transition_to_release_input_screen(remaining)
        else:
            self.current_player.water_storage = remaining
            water_storage: int = self.current_player.water_storage * 100
            text: str = f"{self.current_player.name} の貯水量が"
            text += f"{water_storage} 万㎥になりました。"
            self._display_status_and_log(text)

    def apply_release_effect_for_human_player(self, after_release_amount: int) -> None:
        """人間プレイヤーに「放流」の効果を適用する。

        Args:
            after_release_amount (int): 放流後の貯水量。
        """

        self.human_player.water_storage = after_release_amount
        water_storage: str = f" {self.human_player.water_storage * 100} 万㎥"
        text: str = f"{self.human_player.name} の貯水量が"
        text += f"{water_storage}になりました。"
        self._display_status_and_log(text)
        self._vars.sounds[SoundScene.RECOVERY].play()

    def _output_water_storage_effect(self, player_name: str, card: Card) -> None:
        """貯水量の増減の効果をログと画面に出力する。

        Args:
            player_name (str): 対象となるプレイヤーの名前。
            card (Card): 場に出されたカードのインスタンス。
        """

        amount: str = f" {abs(card.water_increase_amount) * 100} 万㎥"
        postfix: str = (
            "増えました。" if card.water_increase_amount >= 0 else "減りました。"
        )

        text: str = f"{player_name} の貯水量が{amount}{postfix}"
        self._display_status_and_log(text)

    def _apply_cards_in_hand_effect(self, card: Card) -> None:
        """場に出されたカードの効果を手札に適用する。

        Args:
            card (Card): 場に出されたカードのインスタンス。
        """

        if card.affected_player is AffectedPlayer.SELF:
            draw_cards: list[Card] = [
                self.deck.popleft() for _ in range(card.draw_count)
            ]
            self.current_player.cards_in_hand.extend(draw_cards)
            text: str = f"{self.current_player.name} が山札からカードを"
            text += f" {card.draw_count} 枚引きました。"
        elif card.affected_player is AffectedPlayer.ALL_PLAYERS:
            for player in [p for p in self.players if (not p.is_drop_out)]:
                draw_cards: list[Card] = [
                    self.deck.popleft() for _ in range(card.draw_count)
                ]
                player.cards_in_hand.extend(draw_cards)
            text: str = "全員 が山札からカードを"
            text += f" {card.draw_count} 枚ずつ引きました。"
        else:
            text: str = ""

        self._display_status_and_log(text)

    ###############################################################################
    # 失格者の判定

    def get_players_exceeding_range(self) -> list[Player]:
        """貯水量が範囲を超えたプレイヤーの list を返す。

        Returns:
            list[Player]: 貯水量が範囲を超えたプレイヤーの list。
        """

        shifted_players: list[Player] = self._get_shifted_players()
        active_players: list[Player] = [
            p for p in shifted_players if (not p.is_drop_out)
        ]
        capacity: int = constants.WATER_CAPACITY
        return [
            p
            for p in active_players
            if ((p.water_storage <= 0) or (p.water_storage > capacity))
        ]

    def _get_shifted_players(self) -> list[Player]:
        """現在の順番のプレイヤーを先頭にした list を返す。

        Returns:
            list[Player]: 現在の順番のプレイヤーを先頭にした list。
        """

        shifted: list[Player] = self.players
        while shifted[0] is not self.current_player:
            shifted = shifted[1:] + shifted[:1]

        return shifted

    def process_for_player_exceeding_range(
        self, players_exceeding_range: list[Player]
    ) -> Player | None:
        """貯水量が範囲を超えたプレイヤーの判定と処理を行う。

        Args:
            players_exceeding_range (list[Player]): 貯水量が範囲を超えたプレイヤーの list。

        Returns:
            Player | None: 復帰したプレイヤーが存在する場合は、復帰したプレイヤーのインスタンス。
        """

        player: Player = players_exceeding_range[0]
        if player.water_storage <= 0:
            card_type: CardType = CardType.TORRENTIAL_RAIN_S
            return self._check_drop_out(player, is_overflow=False, card_type=card_type)
        elif player.water_storage > constants.WATER_CAPACITY:
            card_type: CardType = CardType.RELEASE
            return self._check_drop_out(player, is_overflow=True, card_type=card_type)

    def _check_drop_out(
        self, player: Player, is_overflow: bool, card_type: CardType
    ) -> Player | None:
        """失格の判定と処理を行う。

        Args:
            player (Player): 対象となるプレイヤー。
            is_overflow (bool): 上限を超えた場合は True。
            card_type (CardType): 復活可能なカードの種類。

        Returns:
            Player | None: 復帰した場合は、復帰したプレイヤーのインスタンス。
        """

        recovery_cards: list[Card] = [
            c for c in player.cards_in_hand if (c.type is card_type)
        ]

        if any(recovery_cards):
            player.cards_in_hand.remove(recovery_cards[0])
            self.cards_on_table.append(recovery_cards[0])
            player.water_storage = 3
            card_name: str = recovery_cards[0].name
            msg: str = f"{player.name} は「{card_name}」を出して復帰しました。"
            self._display_status_and_log(msg)
            self._vars.sounds[SoundScene.RECOVERY].play()
            return player
        else:
            player.is_drop_out = True
            player.index = -1
            msg: str = f"{player.name} は{'越水' if is_overflow else '渇水'}"
            msg += "で失格になりました。"
            self._display_status_and_log(msg)
            self._vars.sounds[SoundScene.DROP_OUT].play()
            self._return_hand_to_deck()
            self._set_player_indexes()
            self.players = self._set_prev_and_next_player(self.players)
            return None

    def process_drop_out_players_after_game_end(self) -> None:
        """ゲーム終了後に失格者の処理を行う。"""

        active_players: list[Player] = [p for p in self.players if (not p.is_drop_out)]
        for player in active_players:
            if player.water_storage <= 0:
                player.is_drop_out = True
                player.index = -1
                msg: str = f"{player.name} は渇水で失格になりました。"
                self._display_status_and_log(msg)
            elif player.water_storage > constants.WATER_CAPACITY:
                player.is_drop_out = True
                player.index = -1
                msg: str = f"{player.name} は越水で失格になりました。"
                self._display_status_and_log(msg)

    def _return_hand_to_deck(self) -> None:
        """失格したプレイヤーの手札をシャッフルして山札に戻す。"""

        for player in self.players:
            if player.is_drop_out and any(player.cards_in_hand):
                random.shuffle(player.cards_in_hand)
                self.deck.extend(player.cards_in_hand)
                player.cards_in_hand = []

    ###############################################################################
    # ゲーム終了の判定

    def has_players_without_hand(self) -> bool:
        """手札がなくなったプレイヤーがいるかどうかを返す。

        Returns:
            bool: 手札がなくなったプレイヤーがいる場合は True。
        """

        active_players: list[Player] = [p for p in self.players if (not p.is_drop_out)]
        empty_hand_players: list[Player] = [
            p for p in active_players if (not any(p.cards_in_hand))
        ]
        if any(empty_hand_players):
            text: str = f"{empty_hand_players[0].name} の手札がなくなりました。"
            self._display_status_and_log(text)

        return any(empty_hand_players)

    def is_game_end_by_drop_out(self) -> bool:
        """
        人間プレイヤー、すべてのプレイヤー、またはひとりを残して他のプレイヤーが
        全員失格したことによるゲーム終了の判定を行う。

        Returns:
            bool: ゲーム終了の場合は True。
        """

        if not any(self.get_players_exceeding_range()):
            active_players: list[Player] = [
                p for p in self.players if (not p.is_drop_out)
            ]

            if not any(active_players):
                text: str = "すべてのプレイヤーが失格になりました。"
                self._display_status_and_log(text)
                return True
            elif self.human_player.is_drop_out:
                text: str = "あなた は失格になりました。"
                self._display_status_and_log(text)
                return True
            elif len(active_players) == 1:
                text: str = f"{active_players[0].name}"
                text += " 以外のすべてのプレイヤーが失格になりました。"
                self._display_status_and_log(text)
                return True
            else:
                return False
        else:
            return False

    ###############################################################################
    # 対戦成績の記録

    def record_result(self) -> None:
        """対戦成績を記録する。"""

        results: list[constants.ResultDict] = self._config.results
        player_count = self._config.player_count
        difficulty: Difficulty = self._config.difficulty
        mode: str = difficulty.get_mode_name(player_count)
        result: constants.ResultDict = [d for d in results if (d["mode"] == mode)][0]

        result["play_count"] = int(result["play_count"]) + 1

        if self.human_player.is_win:
            result["win_count"] = int(result["win_count"]) + 1

        if self.human_player.is_draw:
            result["draw_count"] = int(result["draw_count"]) + 1

        if self.human_player.is_drop_out:
            result["dropout_count"] = int(result["dropout_count"]) + 1

    ###############################################################################
    # ログ等への出力

    def log_starting_hand(
        self, player: Player, is_before: bool, exchanged_count: int = 0
    ) -> None:
        """指定されたプレイヤーのカード交換前と交換後の手札の一覧をログに出力する。

        Args:
            player (Player): 対象となるプレイヤー。
            is_before (bool): カード交換前の場合は True。
            exchanged_count (int, optional): 交換された枚数（デフォルトは 0）。
        """

        text: str = f"{player.name} の交換{('前' if is_before else '後')}"
        text += "の手札： "
        in_hand: list[Card] = player.cards_in_hand
        contents: list[str] = [c.name for c in sorted(in_hand, key=lambda c: c.type)]
        text += "[ " + " | ".join(contents) + " ]"
        self._logger.info(text)

        if not is_before:
            text = f"{player.name} は {exchanged_count} 枚交換しました。"
            self._logger.info(text)

    def log_players_status(self, is_result: bool = False) -> None:
        """すべてのプレイヤーの現在の状況をログに出力する。

        Args:
            is_result (bool, optional): 結果を出力する場合は True。
        """

        for player in self.players:
            in_hand: list[Card] = sorted(player.cards_in_hand, key=lambda c: c.type)
            if any(in_hand):
                str_in_hand: str = " | ".join([c.name for c in in_hand])
                str_in_hand = f"[ {str_in_hand} ]"
            else:
                str_in_hand: str = ""

            text: str = ("［結果］" if is_result else "［状況］") + player.name
            text += " <<失格>>" if player.is_drop_out else ""
            text += f"　貯水量：{(player.water_storage * 100):4} 万㎥　"
            text += f"手札：{len(in_hand):2} 枚　{str_in_hand}"
            self._logger.info(text)

    def log_card_count(self) -> None:
        """手札・場札・山札の枚数をログに出力する。"""

        in_hands: int = sum([len(p.cards_in_hand) for p in self.players])
        deck: int = len(self.deck)
        on_table: int = len(self.cards_on_table)

        text: str = f"各カードの枚数　　手札の合計：{in_hands:2} 枚"
        text += f"　　場札：{on_table:2} 枚　　山札：{deck:2} 枚"
        text += f"　（合計：{(in_hands + deck + on_table):2} 枚）"
        self._logger.info(text)

    def log_current_player(self) -> None:
        """現在のプレイヤーをログに出力する。

        Args:
            current_player (Player): 現在の順番のプレイヤー。
        """

        text: str = f"{self.current_player.name} の順番です。"
        self._logger.info(text)

    def output_winner(self) -> None:
        """ログとゲーム状況の表示に優勝者を出力する。"""

        active_players: list[Player] = [p for p in self.players if (not p.is_drop_out)]
        if not any(active_players):
            text: str = "★ すべてのプレイヤーが失格になったのでゲーム終了です。"
        elif self.human_player.is_drop_out:
            text: str = "★ あなた が失格になったのでゲーム終了です。"
        elif len(active_players) == 1:
            text: str = f"★ {active_players[0].name} の優勝です。"
            active_players[0].is_win = True
        else:
            largest: int = max([p.water_storage for p in active_players])
            winners: list[Player] = [
                p for p in active_players if p.water_storage == largest
            ]

            if len(winners) == self._config.player_count:
                text: str = "★ 引き分けです。"
                for player in winners:
                    player.is_draw = True
            else:
                names: str = "、".join([p.name for p in winners])
                text: str = f"★ {names} の優勝です。"
                for player in winners:
                    player.is_win = True

        self._display_status_and_log(text, True)

    def _display_status_and_log(self, text: str, insert_lf: bool = False) -> None:
        """ログとゲーム状況の表示にテキストを出力する。

        Args:
            text (str): 出力するテキスト。
            insert_lf (bool, optional): 改行を挿入する場合は True。
        """

        self._logger.info(text)

        text = f"\n{text}" if insert_lf else text
        self._add_status(text)
