"""
メインモジュール

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import copy
import logging
import logging.handlers
import os
import sys
import time

if hasattr(sys, "_MEIPASS"):
    os.environ["KIVY_NO_CONSOLELOG"] = "1"

from kivy.config import Config

Config.set("graphics", "width", 800)
Config.set("graphics", "height", 600)
Config.set("graphics", "minimum_width", 800)
Config.set("graphics", "minimum_height", 600)
Config.set("graphics", "resizable", False)
Config.set("input", "mouse", "mouse, disable_multitouch")
Config.set("kivy", "exit_on_escape", "0")
# Config.set("modules", "showborder", "")

import kivy  # noqa
import kivy.resources  # noqa
from kivy.app import App  # noqa
from kivy.core.text import DEFAULT_FONT, LabelBase  # noqa

import constants  # noqa
import views.root_widget as root_widget  # noqa
from models.difficulty import Difficulty  # noqa
from models.my_config import MyConfig  # noqa
from models.shared_vars import SharedVars  # noqa
from my_utilities import helper  # noqa
from my_utilities import json_io  # noqa


class MainApp(App):
    use_kivy_settings = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        kivy.resources.resource_add_path(helper.get_resource_path("fonts"))
        kivy.resources.resource_add_path(helper.get_resource_path("images"))
        kivy.resources.resource_add_path(helper.get_resource_path("sounds"))

        LabelBase.register(DEFAULT_FONT, "VL-PGothic-Regular.ttf")

        self.title = constants.PRODUCT_NAME
        self.icon = "favicon.ico"

        self._logger: logging.Logger = self._initialize_logger()
        self._shared_vars: SharedVars = SharedVars()
        self._config: MyConfig = self._get_config_values()
        self._config.difficulty = Difficulty(self._config.difficulty)

    def build(self):
        root = root_widget.RootWidget(self._config, self._shared_vars, self._logger)
        return root

    def on_stop(self):
        """アプリを終了したとき。"""

        self._output_config_values()

        self._logger.info("======== アプリを終了しました ========")

    def _initialize_logger(self) -> logging.Logger:
        """logging.Logger のインスタンスを作成し、初期設定して返す。

        Returns:
            logging.Logger: 初期設定された logging.Logger のインスタンス。
        """

        os.makedirs(constants.LOG_DIR_PATH, exist_ok=True)

        logger: logging.Logger = logging.getLogger(__name__)
        logger.setLevel(logging.INFO)

        file_handler = logging.handlers.RotatingFileHandler(
            f"{constants.LOG_DIR_PATH}/{constants.BASE_NAME}.log",
            encoding="utf-8",
            maxBytes=1_048_576,
            backupCount=10,
        )
        format: str = "%(asctime)s [%(levelname)s] %(message)s"
        file_handler.setFormatter(logging.Formatter(format))
        logger.addHandler(file_handler)

        logger.info(f"========「{constants.PRODUCT_NAME}」を起動しました ========")
        logger.info(constants.VERSION)
        logger.info(constants.COPYRIGHT)

        return logger

    def _get_config_values(self) -> MyConfig:
        """
        設定ファイルからアプリの設定値を読み込み、MyConfig のインスタンスに
        格納して返す。ファイルが存在しない場合は、既定値が格納された MyConfig の
        インスタンスを返す。

        Returns:
            MyConfig: 設定値が格納された MyConfig のインスタンス。
        """

        start_time: float = time.perf_counter()
        self._logger.info("設定ファイルの読み込みを開始しました。")

        file_path: str = constants.CONF_FILE_PATH
        filename: str = os.path.basename(file_path)

        try:
            my_config: MyConfig = json_io.load_dataclass(MyConfig(), file_path)

        except FileNotFoundError:
            self._logger.info(
                f"設定ファイル（{filename}）がありません。既定値を設定します。"
            )
            my_config = MyConfig()
            my_config.results = copy.deepcopy(constants.INITIAL_RESULTS)
            return my_config
        except Exception as ex:
            msg: str = f"設定ファイル（{filename}）を読み込めません。"
            msg += f"既定値を設定します。\n{ex}"
            self._logger.warning(msg)
            my_config = MyConfig()
            my_config.results = copy.deepcopy(constants.INITIAL_RESULTS)
            return my_config
        else:
            msg: str = f"設定ファイル（{filename}）を読み込みました。"
            self._log_finish_message(msg, start_time)
            return my_config

    def _output_config_values(self) -> None:
        """アプリの設定値を設定ファイルに出力する。"""

        start_time: float = time.perf_counter()
        self._logger.info("設定ファイルの出力を開始しました。")

        file_path: str = constants.CONF_FILE_PATH
        filename: str = os.path.basename(file_path)

        try:
            json_io.output_dataclass(self._config, file_path)

        except PermissionError as ex:
            self._logger.warning(
                f"設定ファイル（{filename}）への書き込み権限がありません。\n{ex}"
            )
        except Exception as ex:
            self._logger.warning(f"設定ファイル（{filename}）が出力できません。\n{ex}")
        else:
            msg: str = f"設定ファイル（{filename}）を出力しました。"
            self._log_finish_message(msg, start_time)

    def _log_finish_message(self, msg: str, start_time: float) -> None:
        """処理の完了メッセージをログに出力する。

        Args:
            msg (str): メッセージ。
            start_time (float): 処理開始時点の時刻。
        """

        elapsed: float = time.perf_counter() - start_time
        msg += f" [{helper.timespan_to_string(elapsed)}]"
        self._logger.info(msg)


if __name__ == "__main__":
    MainApp().run()
