"""
「ふつう」モード時のコンピューターの戦術。

    「だむぽん！」
    Copyright © 2022 toshifumi tsutsui
    Released under the MIT license
    https://wpandora8.net/the_mit_license.html
"""

import logging
import math
import random

import constants
import controllers.tactics.to_exchange_cards as to_exchange_cards
import controllers.tactics.to_play_card as to_play_card
from controllers.tactics.common_tactics import CommonTactics
from models.card import Card
from models.card_type import CardType
from models.player import Player


class InNormalMode:
    """「ふつう」モード時のコンピューターの戦術。"""

    __slots__ = ["_logger", "_ct"]

    def __init__(self, logger: logging.Logger) -> None:
        """「ふつう」モード時のコンピューターの戦術。

        Args:
            logger (logging.Logger): logging.Logger のインスタンス。
        """

        self._logger: logging.Logger = logger
        """logging.Logger のインスタンス"""

        self._ct: CommonTactics = CommonTactics(self._logger)
        """各モード共通のコンピューターの戦術"""

    def select_cards_to_exchange(self, player: Player) -> list[Card]:
        """「ふつう」モードのときに、交換する手札を選択して、その手札の list を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            list[Card]: 手札の list。
        """

        cards: list[Card] = player.cards_in_hand

        types: dict[CardType, int] = {
            CardType.RELEASE: 2,
            CardType.TORRENTIAL_RAIN_S: 1,
            CardType.DRAW_OWN_2_CARDS: 1,
        }

        return to_exchange_cards.leave_target_cards(cards, types)

    def select_card_to_play(self, player: Player, players: list[Player]) -> Card:
        """「ふつう」モードのときに、場に出す手札を選択して返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            Card: 場に出すカード。
        """

        in_hand: list[Card] = player.cards_in_hand

        if self._does_case_01_hold(player, players):
            types: list[CardType] = [
                CardType.RELEASE,
                CardType.DROUGHT_A,
                CardType.DROUGHT_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_02_hold(player):
            types: list[CardType] = [
                CardType.HEAVY_RAIN,
                CardType.RAINY_WEATHER,
                CardType.TORRENTIAL_RAIN_S,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_03_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_04_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_05_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_A]
            )
        elif self._does_case_06_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_07_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(in_hand, [CardType.TYPHOON])
        elif self._does_case_08_hold(player):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_N,
                CardType.THUNDERSTORM_N,
                CardType.TORRENTIAL_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_09_hold(player, players):
            types: list[CardType] = [
                CardType.SHOWER_RAIN_L,
                CardType.THUNDERSTORM_L,
                CardType.TORRENTIAL_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_10_hold(player, players):
            types: list[CardType] = [CardType.RAINY_WEATHER, CardType.HEAVY_RAIN]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_11_hold(player):
            types: list[CardType] = [
                CardType.THUNDERSTORM_N,
                CardType.TORRENTIAL_RAIN_N,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_12_hold(player, players):
            types: list[CardType] = [
                CardType.THUNDERSTORM_L,
                CardType.TORRENTIAL_RAIN_L,
            ]
            card: Card = to_play_card.get_card_by_priority(in_hand, types)
        elif self._does_case_13_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.HEAVY_RAIN]
            )
        elif self._does_case_14_hold(player):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_N]
            )
        elif self._does_case_15_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_L]
            )
        elif self._does_case_16_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.DROUGHT_L]
            )
        elif self._does_case_17_hold(player, players):
            card: Card = to_play_card.get_card_by_priority(
                in_hand, [CardType.TORRENTIAL_RAIN_S]
            )
        else:
            water_storage: int = player.water_storage
            is_largest: bool = self._ct.is_largest_water_amount(player, players)
            other_cards: list[Card] = to_play_card.get_cards_in_other_cases(
                in_hand, water_storage, is_largest
            )
            if any(other_cards):
                text: str = f"{player.name}の戦術："
                self._logger.info(f"{text}「その他のケース」に該当しました。")

            else:
                text: str = f"{player.name}の戦術："
                self._logger.info(f"{text} どのケースにも該当しませんでした。")

            card: Card = random.choice(other_cards if any(other_cards) else in_hand)

        return card

    def _does_case_01_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量がトップで、貯水量がしきい値を超えており、かつ、「放流」か
        「日照り（全）」か「日照り（多）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        def get_threshold() -> int:
            """（関数内）貯水量を減らすカードを出す貯水量のしきい値をランダムに算出して返す。"""

            threshold: int = int(round(random.normalvariate(9.0, 2.0), 0))

            lower: int = 2
            upper: int = constants.WATER_CAPACITY

            return (
                lower
                if (threshold < lower)
                else (upper if (threshold > upper) else threshold)
            )

        cases: list[bool] = []
        cases.append(self._ct.is_largest_water_amount(player, players))
        cases.append(player.water_storage >= get_threshold())

        types: list[CardType] = [
            CardType.RELEASE,
            CardType.DROUGHT_A,
            CardType.DROUGHT_L,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量がトップで、貯水量がしきい値を"
            text += "超えており、かつ、「放流」か「日照り（全）」か「日照り（多）」"
            text += "のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_02_hold(self, player: Player) -> bool:
        """対象プレイヤーの貯水量が 2 以下で、「雨」か「大雨」か「集中豪雨（自）」の
        いずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.water_storage <= 2)

        types: list[CardType] = [
            CardType.HEAVY_RAIN,
            CardType.RAINY_WEATHER,
            CardType.TORRENTIAL_RAIN_S,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 2 以下で、「雨」か「大雨」か"
            text += "「集中豪雨（自）」のいずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_03_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤーの貯水量が 1 で、最も多い貯水量が 2、かつ、
        「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.water_storage == 1)

        largest_amount: int = self._ct.get_largest_water_amount(players)
        cases.append(largest_amount == 2)
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.DROUGHT_L])
        )

        if all(cases):
            text: str = "対象プレイヤーの貯水量が 1 で、最も多い貯水量が 2、かつ、"
            text += "「日照り（多）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_04_hold(self, player: Player, players: list[Player]) -> bool:
        """失格になっていないプレイヤーの 70% 以上の人数のプレイヤーが貯水量 2 以下で、
        対象プレイヤーの貯水量は 3 以上、かつ、「日照り（全）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        lowests: list[Player] = self._ct.get_players_of_lowest_water_amount(players)
        cases.append(lowests[0].water_storage <= 2)
        cases.append(len(lowests) >= math.floor(len(players) * 0.70))
        cases.append(player.water_storage >= 3)
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.DROUGHT_A])
        )

        if all(cases):
            text: str = "失格になっていないプレイヤーの 70% 以上の人数のプレイヤー"
            text += "が貯水量 2 以下で、対象プレイヤーの貯水量は 3 以上、かつ、"
            text += "「日照り（全）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_05_hold(self, player: Player, players: list[Player]) -> bool:
        """失格になっていないプレイヤーの 70% 以上の人数のプレイヤーが貯水量 2 以下で、
        対象プレイヤーの貯水量も 2 以下、かつ、「日照り（全）」と「集中豪雨（自）」カードを
        持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        number_of_target: int = len([p for p in players if (p.water_storage <= 2)])
        cases.append(number_of_target >= math.floor(len(players) * 0.70))
        cases.append(player.water_storage <= 2)
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.DROUGHT_A])
        )
        cases.append(
            self._ct.has_specified_card(
                player.cards_in_hand, [CardType.TORRENTIAL_RAIN_S]
            )
        )

        if all(cases):
            text: str = "失格になっていないプレイヤーの 70% 以上の人数のプレイヤー"
            text += "が貯水量 2 以下で、対象プレイヤーの貯水量も 2 以下、かつ、"
            text += "「日照り（全）」と「集中豪雨（自）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_06_hold(self, player: Player, players: list[Player]) -> bool:
        """失格になっていないプレイヤーの 70% 以上の人数のプレイヤーが貯水量 9 以上の最少で、
        対象プレイヤーの貯水量は最少ではなく、かつ、「台風（少）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        lowests: list[Player] = self._ct.get_players_of_lowest_water_amount(players)
        cases.append(lowests[0].water_storage >= 9)
        cases.append(len(lowests) >= math.floor(len(players) * 0.70))
        cases.append(not self._ct.is_lowest_water_amount(player, players))
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.TYPHOON])
        )

        if all(cases):
            text: str = "失格になっていないプレイヤーの 70% 以上の人数のプレイヤー"
            text += "が貯水量 9 以上の最少で、対象プレイヤーの貯水量は最少"
            text += "ではなく、かつ、「台風（少）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_07_hold(self, player: Player, players: list[Player]) -> bool:
        """失格になっていないプレイヤーの 70% 以上の人数のプレイヤーが貯水量 9 以上の最少で、
        かつ、「台風（少）」と「放流」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        lowests: list[Player] = self._ct.get_players_of_lowest_water_amount(players)
        cases.append(lowests[0].water_storage >= 9)
        cases.append(len(lowests) >= math.floor(len(players) * 0.70))
        cases.append(not self._ct.is_lowest_water_amount(player, players))

        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.TYPHOON])
        )
        cases.append(
            self._ct.has_specified_card(player.cards_in_hand, [CardType.RELEASE])
        )

        if all(cases):
            text: str = "失格になっていないプレイヤーの 70% 以上の人数のプレイヤー"
            text += "が貯水量 9 以上の最少で、かつ、「台風（少）」と「放流」"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_08_hold(self, player: Player) -> bool:
        """次の順番のプレイヤーの貯水量が 15 のとき、対象プレイヤーが「集中豪雨（次）」か
        「雷雨（次）」か「にわか雨（次）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.next_player.water_storage == 15)  # type: ignore

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_N,
            CardType.THUNDERSTORM_N,
            CardType.SHOWER_RAIN_N,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "次の順番のプレイヤーの貯水量が 15 のとき、対象プレイヤー"
            text += "が「集中豪雨（次）」か「雷雨（次）」か「にわか雨（次）」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_09_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤー以外に貯水量が 15 のプレイヤーが存在し、対象プレイヤーの貯水量はトップではなく、
        かつ、「集中豪雨（多）」か「雷雨（多）」か「にわか雨（多）」のいずれかのカードを
        持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(largest == 15)
        cases.append(player.water_storage < largest)

        types: list[CardType] = [
            CardType.TORRENTIAL_RAIN_L,
            CardType.THUNDERSTORM_L,
            CardType.SHOWER_RAIN_L,
        ]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤー以外に貯水量が 15 のプレイヤーが存在し、"
            text += "対象プレイヤーの貯水量はトップではなく、かつ、"
            text += "「集中豪雨（多）」か「雷雨（多）」か「にわか雨（多）」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_10_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤー以外に貯水量が 15 のプレイヤーが存在し、対象プレイヤーの貯水量は 9 以下で、
        かつ、「大雨」か「雨」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(self._ct.get_largest_water_amount(players) == 15)
        cases.append(player.water_storage <= 9)

        types: list[CardType] = [CardType.RAINY_WEATHER, CardType.HEAVY_RAIN]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤー以外に貯水量が 15 のプレイヤーが存在し、"
            text += "対象プレイヤーの貯水量は 9 以下で、かつ、「大雨」か「雨」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_11_hold(self, player: Player) -> bool:
        """次の順番のプレイヤーの貯水量が 13 以上で、かつ、「集中豪雨（次）」か
        「雷雨（次）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []
        cases.append(player.next_player.water_storage >= 13)  # type: ignore

        types: list[CardType] = [CardType.TORRENTIAL_RAIN_N, CardType.THUNDERSTORM_N]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "次の順番のプレイヤーの貯水量が 13 以上で、かつ、"
            text += "「集中豪雨（次）」か「雷雨（次）」の"
            text += "いずれかのカードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_12_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤー以外に貯水量が 13 以上のプレイヤーが存在し、対象プレイヤーの貯水量はトップではなく、
        かつ、「集中豪雨（多）」か「雷雨（多）」のいずれかのカードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        cases: list[bool] = []

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(largest >= 13)
        cases.append(player.water_storage < largest)

        types: list[CardType] = [CardType.TORRENTIAL_RAIN_L, CardType.THUNDERSTORM_L]
        cases.append(self._ct.has_specified_card(player.cards_in_hand, types))

        if all(cases):
            text: str = "対象プレイヤー以外に貯水量が 13 以上のプレイヤーが存在し、"
            text += "対象プレイヤーの貯水量はトップではなく、かつ、"
            text += "「集中豪雨（多）」か「雷雨（多）」のいずれかの"
            text += "カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_13_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤー以外に貯水量が 13 以上のプレイヤーが存在し、対象プレイヤーの貯水量は 7 以下で、
        かつ、「大雨」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.get_largest_water_amount(players) >= 13)
        cases.append(player.water_storage <= 7)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.HEAVY_RAIN]))

        if all(cases):
            text: str = "対象プレイヤー以外に貯水量が 13 以上のプレイヤーが存在し、"
            text += "対象プレイヤーの貯水量は 7 以下で、かつ、"
            text += "「大雨」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_14_hold(self, player: Player) -> bool:
        """次の順番のプレイヤーの貯水量が 11 以上で、かつ、
        「集中豪雨（次）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(player.next_player.water_storage >= 11)  # type: ignore
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_N]))

        if all(cases):
            text: str = "次の順番のプレイヤーの貯水量が 11 以上で、かつ、"
            text += "「集中豪雨（次）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_15_hold(self, player: Player, players: list[Player]) -> bool:
        """対象プレイヤー以外に貯水量が 11 以上のプレイヤーが存在し、対象プレイヤーの貯水量はトップではなく、
        かつ、「集中豪雨（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []

        largest: int = self._ct.get_largest_water_amount(players)
        cases.append(largest >= 11)
        cases.append(player.water_storage < largest)
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_L]))

        if all(cases):
            text: str = "対象プレイヤー以外に貯水量が 11 以上のプレイヤーが存在し、"
            text += "対象プレイヤーの貯水量はトップではなく、かつ、"
            text += "「集中豪雨（多）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_16_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの貯水量はトップではなく、
        かつ「日照り（多）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append(not self._ct.is_largest_water_amount(player, players))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.DROUGHT_L]))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの貯水量は"
            text += "トップではなく、かつ「日照り（多）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)

    def _does_case_17_hold(self, player: Player, players: list[Player]) -> bool:
        """ゲームの終盤において、対象プレイヤーの貯水量が 5 以下のとき、または、10 以下でも貯水量が
        5 増えればトップに立てるとき、「集中豪雨（自）」カードを持っていれば True を返す。

        Args:
            player (Player): 対象プレイヤー。
            players (list[Player]): 失格になっていないすべてのプレイヤー。

        Returns:
            bool: 条件に該当すれば True。
        """

        in_hand: list[Card] = player.cards_in_hand
        amount: int = player.water_storage
        largest: int = self._ct.get_largest_water_amount(players)

        cases: list[bool] = []
        cases.append(self._ct.is_late_stage(players))
        cases.append((amount <= 5) or ((amount <= 10) and (amount + 5 >= largest)))
        cases.append(self._ct.has_specified_card(in_hand, [CardType.TORRENTIAL_RAIN_S]))

        if all(cases):
            text: str = "ゲームの終盤において、対象プレイヤーの貯水量が 5 以下の"
            text += "とき、または、10 以下でも貯水量が 5 増えればトップに立てる"
            text += "とき、「集中豪雨（自）」カードを持っている"
            self._ct.output_case_to_log(player, text)

        return all(cases)
